async function submitUserMessage(event) {
    event.preventDefault();
    try {
        const userMessage = document.getElementById('user_message').value;
        await askBotPipeline(userMessage);
    } catch (error) {
        await handleBotResponse("An error occurred: " + error.stack); // show the error to the user
    }
}



function showSelectedConnectionTypeFields() {
    // Hide all input groups and remove 'required' attribute from their input elements
    document.querySelectorAll('.input-group').forEach(group => {
        group.style.display = 'none';
        group.querySelectorAll('input, textarea').forEach(input => {
            input.removeAttribute('required');
        });
    });

    // Show the selected input group and add 'required' attribute to its input elements 
    // (except for 'Additional Connection String Options' field)
    const selectedValue = document.getElementById('ConnectionTypesDropdown').value;
    if (selectedValue) {
        const selectedGroup = document.getElementById(selectedValue);
        selectedGroup.style.display = 'flex';
        selectedGroup.querySelectorAll('input, textarea').forEach(input => {
            if (!input.name.includes('additional_options')) {
                input.setAttribute('required', '');
            }
        });
    }

    // clear any prior messages
    const responseElement = document.getElementById('responseAddConnection');
    responseElement.innerText = '';
}



async function addDBConnection(event) {
    
    event.preventDefault();
    const responseElement = document.getElementById('responseAddConnection');
    const spinner = document.getElementById('spinnerDatabaseAdd');
    const form = document.getElementById('AddConnectionForm');
    const formData = new FormData(form);

    responseElement.innerText = '';  // clear previous messages
    spinner.style.display = 'block';  // Show the spinner

    try {
        const res = await fetch('/manage-db-connection', {
            method: 'POST',
            body: formData
        });
        
        spinner.style.display = 'none';  // Hide the spinner

        const response = await res.json();
        
        const responseMsg = response.message;
        responseElement.innerText = responseMsg;
        if (responseMsg.startsWith('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }

        // add connection to list of manually-added connections
        if (!responseMsg.startsWith('Error')) {
            const dbId = response.db_id;
            const dbEngine = formData.get('connectionDropdown');
            const dbDisplayName = formData.get(`display_name_${dbEngine}`);
            
            const connectionRow = document.createElement('tr');
            connectionRow.id = `tr_db_ref_${dbId}`;
            connectionRow.innerHTML = `
                <tr id="tr_db_ref_${dbId}">
                    <td><a href="#" id="td_db_ref_${dbId}" onclick="openTabNoPreloadDatabaseConnection(${dbId})">${dbDisplayName}</a></td>
                    <td>${dbEngine}</td>
                </tr>
            `;
            document.getElementById('manualConnectionsTable').appendChild(connectionRow);
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = 'Error submitting form';
        responseElement.style.color = 'red'; 
    }
}



// Function to scroll to the bottom
async function scrollToBottom() {

    // in full-screen mode, scroll is on content div
    const contentDiv = document.getElementById('content');
    contentDiv.scrollTo({
        top: contentDiv.scrollHeight,
        behavior: 'smooth'
    });

    // in chatbot+dashboard mode, scroll is on conversation div
    const conversationtDiv = document.getElementById('conversation');
    conversationtDiv.scrollTo({
        top: conversationtDiv.scrollHeight,
        behavior: 'smooth'
    });
}



async function syncDBs() {

    const spinner = document.getElementById('spinnerSyncDBs');
    const responseElement = document.getElementById('responseSyncDBs');
    const syncButton = document.getElementById('sync_with_bi_tool');
    syncButton.disabled = true;  // Disable the button to prevent multiple clicks
    syncButton.style.cursor = 'not-allowed';  // Change cursor to indicate button is disabled
    responseElement.innerText = '';  // clear previous messages
    spinner.style.display = 'block';

    try {
        const response = await fetch('/sync-dbs', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        }).catch(error => console.error('Error:', error)); 
        const res = await response.json();
        const result = res['response'];
        const res_type = res['response_type'];
        
        spinner.style.display = 'none';
        syncButton.disabled = false;  // Re-enable the button
        syncButton.style.cursor = '';  // Reset cursor

        if (res_type == 'list') {
            showNotification(get_translation('Sync completed.', language), true);
 
            html_content = `<div style="padding: 0px 20px;">
                <table>
                    <tr>
                        <th>Name</th>
                        <th>Engine</th>
                    </tr>`;
            result.forEach(db => {
                html_content += `
                    <tr id="tr_db_ref_${db[0]}">
                        <td><a href="#" id="td_db_ref_${db[0]}" onclick="openTabNoPreloadDatabaseConnection(${db[0]})">${db[2]}</a></td>
                        <td>${db[1]}</td>
                    </tr>`;
            });
            html_content += `</table></div>`;
            responseElement.innerHTML = html_content;
        } else {
            if (result.startsWith('Error')) {
                responseElement.style.color = 'red';
            }
            if (result.startsWith('No database connection found')) {
                responseElement.style.color = 'blue';
            }
            responseElement.innerText = result;
        }

    } catch (error) {
        spinner.style.display = 'none';
        syncButton.disabled = false;  // Re-enable the button
        syncButton.style.cursor = '';  // Reset cursor
        responseElement.style.color = 'red'; 
        responseElement.innerText = `Sorry. Something Went Wrong (${error})`;
    }
}


async function refreshManualDBs() {

    const spinner = document.getElementById('spinnerRefreshManualDBs');
    const responseElement = document.getElementById('responseRefreshManualDBs');
    const refreshButton = document.getElementById('refresh_manual_dbs_btn');
    refreshButton.disabled = true;  // Disable the button to prevent multiple clicks
    refreshButton.style.cursor = 'not-allowed';  // Change cursor to indicate button is disabled
    responseElement.innerText = '';  // clear previous messages
    spinner.style.visibility = 'visible';

    try {
        const response = await fetch('/refresh-manually-added-dbs', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        }).catch(error => console.error('Error:', error)); 
        const res = await response.json();
        const result = res['response'];
        
        spinner.style.visibility = 'hidden';
        refreshButton.disabled = false;  // Re-enable the button
        refreshButton.style.cursor = '';  // Reset cursor

        if (result == 'Refresh Completed.') {
            showNotification(get_translation('Refresh completed.', language), true, 3000);
        } else {
            if (result.startsWith('Error')) {
                responseElement.style.display = 'block';
                responseElement.style.color = 'red';
            }
            responseElement.innerText = result;
        }

    } catch (error) {
        spinner.style.visibility = 'hidden';
        refreshButton.disabled = false;  // Re-enable the button
        refreshButton.style.cursor = '';  // Reset cursor
        responseElement.style.display = 'block';
        responseElement.style.color = 'red'; 
        responseElement.innerText = `Sorry. Something Went Wrong (${error})`;
    }
}



async function changePassword(event) {
    event.preventDefault();
    
    const form = document.getElementById('changePasswordForm');
    const formData = new FormData(form);
    
    const responseElement = document.getElementById('responseChangePassword');
    responseElement.innerText = '';
    const spinner = document.getElementById('spinnerChangePassword');
    spinner.style.display = 'block';
    

    try {
        const response = await fetch('/change-password', {
            method: 'POST',
            body: formData
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



async function saveMetadata(event, db_id, table_id) {
    event.preventDefault();
    
    const form = document.getElementById('save_metadata_' + table_id);
    const formData = new FormData(form);
    formData.append('db_id', db_id);
    formData.append('table_id', table_id);
    
    const spinner = document.getElementById('spinnerSaveMetadata_' + table_id);
    spinner.style.display = 'block';
    const responseElement = document.getElementById('responseSaveMetadata_' + table_id);

    try {
        const response = await fetch('/save-metadata', {
            method: 'POST',
            body: formData
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



async function saveQuerySuggestions(dbId, event) {
    event.preventDefault();
    
    const form = document.getElementById('querySuggestionForm');
    const formData = new FormData(form);
    formData.append('dbId', dbId);

    const spinner = document.getElementById('spinnerQuerySuggestionSetup');
    spinner.style.display = 'block';
    const responseElement = document.getElementById('responseQuerySuggestionSetup');

    try {
        const response = await fetch('/save-query-suggestions', {
            method: 'POST',
            body: formData
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



async function saverlac(groupName, dbId, tableId) {
    event.preventDefault();
    const postfix = '_' + groupName + '_' + dbId + '_' + tableId;
    const spinner = document.getElementById('spinnerSaverlac' + postfix);
    spinner.style.display = 'block';
    const responseElement = document.getElementById('responseSaverlac' + postfix);

    const inputElementForFilterValue = document.getElementById("filter_value_" + groupName + '_' + tableId);
    var dropdown = document.getElementById('rlacDropdown_' + groupName + '_' + tableId);
    var selectedOption = dropdown.options[dropdown.selectedIndex];
    var selectedOptionId = selectedOption.id;
    try {
        const response = await fetch('/save-rlac', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({groupName:groupName, dbId:dbId, tableId:tableId, fieldId:selectedOptionId, filterValue:inputElementForFilterValue.value })
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



async function saveAnalyticalModeConfig() {
    event.preventDefault();
    const spinner = document.getElementById('spinnerSaveAnalyticalModeConfig');
    spinner.style.display = 'block';
    const responseElement = document.getElementById('responseAlyticalModeConfig');

    var rowLimitDropdown = document.getElementById('analytical-mode-row-limit-dropdown');
    var rowLimit = rowLimitDropdown.options[rowLimitDropdown.selectedIndex].value;
    var metadata = document.getElementById('analytical-mode-metadata').value;
    const includeExplanation = document.getElementById('toggle-button-include-explanation').checked;

    try {
        const response = await fetch('/save-analytical-mode-config', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                rowLimit:rowLimit, 
                metadata:metadata,
                includeExplanation:includeExplanation
            })
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.style.color = 'red'; 
        responseElement.innerText = `Save Failed. Error: ${error}`;
    }
}


async function savePerformanceTuning() {
    event.preventDefault();
    const spinner = document.getElementById('spinnerSavePerformanceTuning');
    spinner.style.display = 'block';
    const responseElement = document.getElementById('responseSavePerformanceTuning');

    const schemaPruning = document.getElementById('toggle-button-schema-pruning').checked
    var llmModel1Dropdown = document.getElementById('llm-model1-dropdown');
    var llmModel2Dropdown = document.getElementById('llm-model2-dropdown');
    var llmModel3Dropdown = document.getElementById('llm-model3-dropdown');
    var llm1 = llmModel1Dropdown.options[llmModel1Dropdown.selectedIndex].value;
    var llm2 = llmModel2Dropdown.options[llmModel2Dropdown.selectedIndex].value;
    var llm3 = llmModel3Dropdown.options[llmModel3Dropdown.selectedIndex].value;

    try {
        const response = await fetch('/save-performance-tuning', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({llmModel1:llm1, llmModel2:llm2, llmModel3:llm3, schemaPruningEnabled:schemaPruning })
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



function openPopupScreen(elementId, refId='', groupName='') {

    document.getElementById(elementId).style.display = "block";

    if (elementId == 'groupCreation') {
        // clear the text field
        var groupName = document.getElementById('userProvidedgroupName');
        groupName.value = '';
    }
    if (elementId == 'deleteGroupConfirmation') {
        groupBtnToDelete = document.getElementById(refId);
    }
    if (elementId == 'removeUserFromGroupConfirmation') {
        userRowToDelete = document.getElementById(refId);
        userToDeleteGroupName = groupName;
    }
    if (elementId == 'renameGroup') {
        groupBtnToRename = document.getElementById(refId);
    }
}

function closePopupScreen(elementId) {
    document.getElementById(elementId).style.display = "none";
}


function showFailedTestError(error) {
    const failedTestScreen = document.getElementById('failed_test_results_popup_screen');
    failedTestScreen.value = error;
    document.getElementById('failed_test_results_popup_screen_div').style.display = 'block';
}


async function createGroup() {
    var groupName = document.getElementById('userProvidedgroupName').value.replace(/_/g, " ");
    var errorDiv = document.getElementById('groupCreationErrorDiv');
    errorDiv.innerHTML = "";

    if (['admins', 'default'].includes(groupName.toLowerCase())){
        errorDiv.innerHTML = "<p style='color: red;'>'Admins' and 'Default' are reserved group names. Please select a different name.</p>"
    } else {
        // add to backend
        const response = await fetch('/modify-groups', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ groupName: groupName, updateType: 'add' }) 
        }).catch(error => console.error('Error:', error)); 

        const result = await response.text();
        if (result.startsWith('Error')) {
            errorDiv.innerHTML = '<p style="color: red;">' + result;
        } else {
            // add to frontend

            // Create the button element
            const button = document.createElement('button');
            button.setAttribute('id', `usergroup_${groupName}`);
            button.className = 'tablinks';
            button.setAttribute('onclick', `openTabNoPreLoadGroupUserList('groupUserList_${groupName}')`);

            // Create the user_groups icon image element
            const img1 = document.createElement('img');
            img1.src = userGroupsIconURL;
            img1.style.paddingRight = '10px';
            
            // Create the arrow image element
            const img2 = document.createElement('img');
            img2.src = arrowForwardIconURL;
            img2.className = 'arrow';

            // Create the text node
            const textNode = document.createTextNode(groupName);

            button.appendChild(img1);
            button.appendChild(textNode);
            button.appendChild(img2);

            const groupsListDiv = document.getElementById('groupsList');
            groupsListDiv.appendChild(button);
            
            // add the inner page
            const innerDiv = document.createElement('div');
            innerDiv.setAttribute('id', `groupUserList_${groupName}`);
            innerDiv.className = 'tabcontent tabAdmin';
            innerDiv.innerHTML = `
                    <button class="back-button" style="max-width: 72px; max-height: 30px; margin-bottom: 50px; margin-top:22px;" onclick="closeTab('groupUserList_${groupName}')"><img src="${arrowBackwardIconURL}" class="back_icon">Back</button>
                    <h3 class="settings_inner_page_title" style="justify-content: center; padding-bottom: 10px;"><img src="${userGroupsIconURL}" style="padding-right: 10px;">${groupName}</h3>
                    
                    <div id="groupUserList_${groupName}_users">
                    </div>
                
                    <div style="margin-bottom: 20px; margin-top: 30px;">
                        <!-- <input value="Rename Group" onclick="openPopupScreen('renameGroup', 'usergroup_${groupName}')" class="intellimenta-color-button" style="text-align: center;"> -->
                        <input value="Delete Group" onclick="openPopupScreen('deleteGroupConfirmation', 'usergroup_${groupName}')" class="intellimenta-color-button" style="background-color: #d82222; text-align:center;">
                    </div>

                    <div id="responseDeleteGroup_${groupName}" style="margin: 20px;"></div>
                    <div id="spinnerDeleteGroup_${groupName}" class="spinner"></div>
            `
            
            userGroupsDiv = document.getElementById('userGroups');
            userGroupsDiv.appendChild(innerDiv);

            // close the popup
            closePopupScreen('groupCreation')   
        }
    }
}


async function deleteGroup() {

    var groupName = groupBtnToDelete.textContent;

    const spinner = document.getElementById(`spinnerDeleteGroup_${groupName}`);
    spinner.style.display = 'block';

    // remove from frontend
    closePopupScreen('deleteGroupConfirmation');
    groupBtnToDelete.style.display = "none";
    
    // remove from backend
    const response = await fetch('/modify-groups', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ groupName: groupName, updateType: 'remove' }) 
    });
    const result = await response.text();
    
    spinner.style.display = 'none';

    if (result.startsWith('Error') ) {
        var errorElement = document.getElementById(`responseDeleteGroup_${groupName}`);
        errorElement.style.color = 'red';
        errorElement.innerText = result;
    } else {
        // close the tab
        closeTab(`groupUserList_${groupName}`);
    }

}


async function renameGroup() {
    oldGroupName = groupBtnToRename.textContent;
    newGroupName = document.getElementById('userProvidedNewgroupName').value;

    //make changes in the backend
    const response = await fetch('/modify-groups', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ groupName: oldGroupName, newGroupName:newGroupName, updateType:'rename' }) 
    }); 
    const result = await response.text();
    
    var responseElement = document.getElementById(`responseDeleteGroup_${oldGroupName}`);

    if (result.startsWith('Error') ) {    
        responseElement.style.color = 'red';
        responseElement.innerText = result;
    } else {
        responseElement.style.color = 'green';
        responseElement.innerText = result;
    }

    closePopupScreen('renameGroup');

}


async function removeUserFromGroup() {
    
    var userEmail = userRowToDelete.cells[0].textContent;
    userRowToDelete.style.display = "none";

    // remove from frontend
    closePopupScreen('removeUserFromGroupConfirmation');

    // remove from backend
    const response = await fetch('/remove-user-from-group', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ userEmail: userEmail, groupName:userToDeleteGroupName }) 
    }).catch(error => console.error('Error:', error)); 

    const result = await response.text();
    if (result.startsWith('Error')) {
        console.error('Failed to remove user from group:', result);
        showNotification('Failed. Check the browser console for more details.', false, 5000);
    }
}


async function createCustomUserAttribute() {
    var attributeName = document.getElementById('userProvidedAttributeName').value;
    var errorDiv = document.getElementById('customUserAttributeCreationErrorDiv');
    errorDiv.innerHTML = "";

    // add to backend
    const response = await fetch('/create-custom-attribute', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ attributeName: attributeName }) 
    }).catch(error => console.error('Error:', error)); 

    const result = await response.text();
    if (result.startsWith('Error')) {
        errorDiv.innerHTML = '<p style="color: red;">' + result;
    } else {
        // add to frontend

        // Create the button element
        const button = document.createElement('button');
        button.setAttribute('id', `userAttribute_${attributeName}`);
        button.className = 'tablinks';
        button.setAttribute('onclick', `openTabNoPreLoadCustomUserAttributeAssignment('${attributeName}')`);

        // Create the icon image element
        const img1 = document.createElement('img');
        img1.src = userAttributeIconURL;
        img1.style.paddingRight = '10px';
        
        // Create the arrow image element
        const img2 = document.createElement('img');
        img2.src = arrowForwardIconURL;
        img2.className = 'arrow';

        // Create the text node
        const textNode = document.createTextNode(attributeName);

        button.appendChild(img1);
        button.appendChild(textNode);
        button.appendChild(img2);
        
        const attributesListDiv = document.getElementById('customUserAttributeList');
        attributesListDiv.appendChild(button);

        // close the popup
        closePopupScreen('customUserAttributeCreation')   
    }

}


async function saveCustomUserAttributeAssignment() {
    
    // show spinner 
    const spinner = document.getElementById('customUserAttributeAssignmentSpinner');
    spinner.style.display = 'block';
    const responseDiv = document.getElementById('customUserAttributeAssignmentResponse');
    const assignmentDiv = document.getElementById('customAttributeAssignmentDiv')

    try {
        // loop over all input elements in the div and get their id and value
        const inputs = assignmentDiv.getElementsByTagName('input');
        const assignmentMapping = {};
        const attributeName = inputs[0].id.split('|')[1];  // the id has the format "attribute|{attribute_name}|{email}"
        for (let i = 0; i < inputs.length; i++) {
            const input = inputs[i];
            const idParts = input.id.split('|');
            const email = idParts[2];
            const value = input.value;
            assignmentMapping[email] = value;
        }
        
        const response = await fetch('/save-attribute-assignment', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ attributeName: attributeName, assignmentMapping: assignmentMapping }) 
        });
        
        spinner.style.display = 'none';  // Hide the spinner
        const result = await response.text();
        
        responseDiv.textContent = result;
        if (result.startsWith('Error')) { responseDiv.style.color = 'red'; }
        else { responseDiv.style.color = 'green'; }
    } catch (error) {
        spinner.style.display = 'none';
        responseDiv.innerText = `An error occured while saving the assignments (${error})`;
        responseDiv.style.color = 'red'; 
    }

}
///////////////////////////////


async function deleteConnection(dbId) {

    const responseElement = document.getElementById('spinnerDeleteConnection');
    const spinner = document.getElementById('responseDeleteConnection');
    spinner.style.display = 'block';
    
    // remove from backend
    const response = await fetch('/delete-db-connection', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ dbId: dbId }) 
    });
    const result = await response.text();
    spinner.style.display = 'none';

    if (result.startsWith('Error') ) {
        responseElement.style.color = 'red';
        responseElement.innerText = result;
    } else {

        // remove the connection table row from frontend
        const connectionRow = document.getElementById(`tr_db_ref_${dbId}`);
        if (connectionRow) {
            connectionRow.style.display = 'none';
        }

        closePopupScreen('deleteConnectionConfirmation');

        // close the tab
        closeTab("db_ref");
    }

}



async function registerActiveDB() {

    const initialLoadingSpinner = document.getElementById('initialLoadingSpinner');
    initialLoadingSpinner.style.display = 'block';
    
    const dropdown = document.getElementById('ActiveDatabaseDropdown');
    const selectedOption = dropdown.options[dropdown.selectedIndex];
    const activeDbId = selectedOption.id;
    const activeDbEngine = selectedOption.getAttribute('engine');
    
    // // show spinner
    // const loadingOverlay = document.getElementById('loading-overlay');  
    // loadingOverlay.style.display = 'block';

    await clearConversationHistory();
    
    const response = await fetch('/save-active-db', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ activeDbId: activeDbId, activeDbEngine:activeDbEngine })
    });
    const result = await response.text();
    if (result != 'Saved.') {
        await handleBotResponse(result);
    }

    // loadingOverlay.style.display = 'none';

    await loadGreetingsAndQuerySuggestions();

    initialLoadingSpinner.style.display = 'none';  // Hide the spinner
}


function getTableMapping(tableId) {
    // Get the table element
    const table = document.getElementById(tableId);
    
    const result = {};

    // Get all rows except the header row
    const rows = Array.from(table.getElementsByTagName('tr')).slice(1);

    // Process each row
    rows.forEach(row => {
        // Get the group name from the first cell
        const groupName = row.getElementsByTagName('td')[0].textContent.trim();
        
        // Get the input value from the second cell
        const input = row.getElementsByTagName('input')[0];
        const value = input ? input.value : '';
        
        // Add to result object
        result[groupName] = value;
    });

    return result;
}


async function saveTwoColConfig() {
    const spinner = document.getElementById('saveTwoColumnSpinner');
    spinner.style.display = 'block';
    const responseDiv = document.getElementById('saveTwoColumnResponse');
    
    try {    
        // get user group assignments
        const userGroupMapping = getTableMapping('two_column_assignment_table');
        const toggleBtnValue = document.getElementById('toggle-button-filter-dashboards-by-user-attributes').checked;

        const response = await fetch('/save-two-column-config', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({userGroupMapping:userGroupMapping, filterDashboardsByUserAttributes:toggleBtnValue})
        });
        spinner.style.display = 'none';  // Hide the spinner
        const result = await response.text();
        
        responseDiv.textContent = result;
        if (result.startsWith('Error')) { responseDiv.style.color = 'red'; }
        else { responseDiv.style.color = 'green'; }
    } catch (error) {
        spinner.style.display = 'none';
        responseDiv.innerText = `An error occured while saving the assignments (${error})`;
        responseDiv.style.color = 'red'; 
    }
}



async function handleEmailSetup(event) {
    event.preventDefault(); 
    
    const responseElement = document.getElementById('responseEmailSetup');
    responseElement.innerText = '';

    const formData = new FormData(document.getElementById('emailSetupForm'));
    const action = event.submitter.value;
    formData.append('action', action);
    
    const spinner = document.getElementById('spinnerEmailSetup');
    spinner.style.display = 'block';  // Show the spinner
    try {
        const response = await fetch('/email-setup', {
            method: 'POST',
            body: formData
        });
        
        spinner.style.display = 'none';  // Hide the spinner
        const result = await response.text();
        
        responseElement.innerText = result;
        if (result.startsWith('Error')) { responseElement.style.color = 'red'; }
        else { responseElement.style.color = 'green'; }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



async function handleConnectBItool(event) {
    event.preventDefault(); 
    
    const responseElement = document.getElementById('responseConnectBItool');
    responseElement.innerText = '';

    const formData = new FormData(document.getElementById('connectBItoolForm'));
    const action = event.submitter.value;
    formData.append('action', action);
    
    const spinner = document.getElementById('spinnerConnectBItool');
    spinner.style.display = 'block';  // Show the spinner
    try {
        const response = await fetch('/connect-bi-tool', {
            method: 'POST',
            body: formData
        });
        
        spinner.style.display = 'none';  // Hide the spinner
        const result = await response.text();
        
        responseElement.innerText = result;
        if (result.startsWith('Error')) { responseElement.style.color = 'red'; }
        else { responseElement.style.color = 'green'; }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while submitting the form (${error})`;
        responseElement.style.color = 'red'; 
    }
}



async function loadIframeInsideDiv(div, iframeSource, iframeType='card') {
    const iframe = document.createElement('iframe');
    iframe.src = iframeSource;
    if (window.innerWidth < 1000 || isChatbotAndDashboards) {
        iframe.style.borderRadius = '0px';
    } else {
        iframe.style.borderRadius = '5px';
    }
    iframe.frameborder= '0px';
    if (iframeType == 'card') {
        iframe.style.width = '99%';
        iframe.style.height = cardHeightPref; //'350px';
    } else {  // dashboard
        iframe.style.width = '100%';
        iframe.style.height = '99%';

        div.innerHTML = `
        <button class="close-button" style="margin:15px; position:absolute" onclick="closePopupScreen('personalDashboard_BG')">
            <img src="${closeIconUrl}" class="close_icon">Close
        </button>
        <button style="margin: auto; cursor: pointer; position: absolute; top: 15px; padding: 5px; width: 120px; left: calc(50% - 60px);" onclick="createTab()">
            Create a Tab
        </button>
        <div id="personalDashboardSpinner" class="spinner" style="display: none; position: absolute; top: 70px; padding: 5px; width: 10px; height: 10px; left: calc(50% - 15px);">
        </div>`;
    }
    div.appendChild(iframe);
}



async function handleBotResponse(botResponse, onPageLoad=false, vizType='', isSingleColor=false, isSessionPage=false) {
    
    const conversationDiv = document.getElementById('conversation');
    
    if ( botResponse != 'NA' ) {
        if (botResponse.startsWith('http')) {
            // If the bot's response is a link, create an iframe to display it
            await loadIframeInsideDiv(conversationDiv, botResponse);
            
            // remove previous action buttons 
            var actionButtonsContainer = document.getElementById('actionButtonsContainer');
            if (actionButtonsContainer) {
                actionButtonsContainer.remove();
            }

            ////// Add Action Buttons //////
            if ( onPageLoad == false && isSessionPage == false ) {  // so action buttons are not shown under personal dashboard or in chat session pages
                
                // *** modify button ***
                if ( ['bar', 'line', 'row', 'combo', 'area', 'scatter', 'waterfall', 'funnel'].includes(vizType) ) {

                    var modifyButton = document.createElement('button');
                    modifyButton.className = 'intellimenta-color-button action-buttons tooltip';
                    modifyButton.style.transform = 'scale(1)';

                    const modifyButtonIcon = document.createElement('img');
                    modifyButtonIcon.src = modifyButtonIconURL;

                    const modifyButtonTooltip = document.createElement('span');
                    modifyButtonTooltip.className = 'tooltiptext';
                    modifyButtonTooltip.style.width = 'auto';
                    modifyButtonTooltip.style.whiteSpace = 'nowrap';
                    modifyButtonTooltip.textContent = get_translation('Modify', language);

                    modifyButton.onclick = function() {
                        modifyButtonTooltip.style.display = 'none';
                        let modifyDiv = document.getElementById('modifyDiv');
                        if (modifyDiv) {
                            modifyDiv.remove();
                        } else {
                            modifyDiv = document.createElement('div');
                            modifyDiv.id = 'modifyDiv';
                            modifyDiv.style.cssText = `
                                display: flex;
                                align-items: flex-start;
                                flex-direction: column;
                                position: absolute;
                                bottom: 100%;
                                left: 0;
                                margin-bottom: 5px;
                                border: 1px solid #ccc;
                                padding: 10px;
                                background-color: #644883;
                                box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.1);
                                border-radius: 5px;
                                width: auto;
                                white-space: nowrap;
                            `;

                            const optionChangeColor = document.createElement('div');
                            optionChangeColor.style.marginBottom = '5px';
                            optionChangeColor.innerHTML = `    
                                <label for="changeColorDropdown">Change Color:</label>
                                <select id="changeColorDropdown" style="max-width: 75px; padding: 2px 0px;">
                                    <option value="" disabled selected>Select Color&nbsp;</option>
                                    <option value="Blue">Blue</option>
                                    <option value="Green">Green</option>
                                    <option value="Yellow">Yellow</option>
                                    <option value="Purple">Purple</option>
                                    <option value="Cyan">Cyan</option>
                                    <option value="Lavender">Lavender</option>
                                    <option value="Orange">Orange</option>
                                    <option value="Red">Red</option>
                                    <option value="Grey">Grey</option>
                                </select>
                            `;

                            const optionRemoveXaxisLabel = document.createElement('label');
                            optionRemoveXaxisLabel.style.display = 'block';
                            optionRemoveXaxisLabel.style.marginBottom = '5px';
                            const checkboxRemoveXaxisLabel = document.createElement('input');
                            checkboxRemoveXaxisLabel.type = 'checkbox';
                            checkboxRemoveXaxisLabel.value = 'Remove X-Axis Label';
                            optionRemoveXaxisLabel.appendChild(checkboxRemoveXaxisLabel);
                            optionRemoveXaxisLabel.appendChild(document.createTextNode(get_translation('Remove X-Axis Label', language)));

                            const optionRemoveYaxisLabel = document.createElement('label');
                            optionRemoveYaxisLabel.style.display = 'block';
                            optionRemoveYaxisLabel.style.marginBottom = '5px';
                            const checkboxRemoveYaxisLabel = document.createElement('input');
                            checkboxRemoveYaxisLabel.type = 'checkbox';
                            checkboxRemoveYaxisLabel.value = 'Remove Y-Axis Label';
                            optionRemoveYaxisLabel.appendChild(checkboxRemoveYaxisLabel);
                            optionRemoveYaxisLabel.appendChild(document.createTextNode(get_translation('Remove Y-Axis Label', language)));

                            const optionHideValues = document.createElement('label');
                            optionHideValues.style.display = 'block';
                            optionHideValues.style.marginBottom = '5px';
                            const checkboxHideValues = document.createElement('input');
                            checkboxHideValues.type = 'checkbox';
                            checkboxHideValues.value = 'Hide Values On Chart';
                            optionHideValues.appendChild(checkboxHideValues);
                            optionHideValues.appendChild(document.createTextNode(get_translation('Hide Values On Chart', language)));

                            const optionRotate = document.createElement('label');
                            optionRotate.style.display = 'block';
                            optionRotate.style.marginBottom = '5px';
                            const checkboxRotate = document.createElement('input');
                            checkboxRotate.type = 'checkbox';
                            checkboxRotate.value = 'Rotate Labels 45°';
                            optionRotate.appendChild(checkboxRotate);
                            optionRotate.appendChild(document.createTextNode(get_translation('Rotate Labels 45°', language)));

                            const submitButton = document.createElement('button');
                            submitButton.textContent = 'Submit';
                            submitButton.style.display = 'block';
                            submitButton.style.margin = 'auto';
                            submitButton.style.marginTop = '3px';
                            submitButton.style.marginBottom = '1px';
                            submitButton.style.cursor = 'pointer';
                            submitButton.onclick = async function(event) {
                                
                                event.stopPropagation();
                                
                                const selectedOptions = [];
                                if (isSingleColor) {
                                    let dropdown = document.getElementById("changeColorDropdown");
                                    let selectedValue = dropdown.value;
                                    if (selectedValue) {selectedOptions.push('change color to ' + selectedValue);}
                                }
                                if (checkboxHideValues.checked) selectedOptions.push(checkboxHideValues.value);
                                if (checkboxRemoveXaxisLabel.checked) selectedOptions.push(checkboxRemoveXaxisLabel.value);
                                if (checkboxRemoveYaxisLabel.checked) selectedOptions.push(checkboxRemoveYaxisLabel.value);
                                if (checkboxRotate.checked) selectedOptions.push(checkboxRotate.value);
                                
                                modifyDiv.remove();
                                
                                if (selectedOptions.length > 0) {
                                    
                                    addToConversation(selectedOptions);
                                    statusUpdatesDiv.innerHTML = '';
                                    loadingOverlay.style.display = 'block';
                                    
                                    for (let option of selectedOptions) {    
                                        var botResponse_ = await askBot(option);
                                    }

                                    loadingOverlay.style.display = 'none';
                                    await handleBotResponse(botResponse_.main_response, false, botResponse_.viz_type, botResponse_.is_single_color);
                                    
                                };

                                // Scroll to the bottom
                                await scrollToBottom();
                            };
                            
                            if (isSingleColor) {
                                modifyDiv.appendChild(optionChangeColor);
                            }
                            modifyDiv.appendChild(optionRemoveXaxisLabel);
                            modifyDiv.appendChild(optionRemoveYaxisLabel);
                            modifyDiv.appendChild(optionRotate);
                            modifyDiv.appendChild(optionHideValues);
                            modifyDiv.appendChild(submitButton);

                            modifyDiv.onclick = function(event) {
                                event.stopPropagation();
                            };

                            modifyButton.appendChild(modifyDiv);
                        }
                    };

                    // Close modifyDiv when clicking outside
                    window.addEventListener('click', function(event) {
                        const modifyDiv = document.getElementById('modifyDiv');
                        if (modifyDiv && !event.target.closest('#modifyDiv') && !event.target.closest('.intellimenta-color-button')) {
                            modifyDiv.remove();
                            modifyButtonTooltip.style.display = 'block';  // enable tooltip again
                        }
                    });

                    modifyButton.appendChild(modifyButtonIcon);
                    modifyButton.appendChild(modifyButtonTooltip);
                }
                

                // *** visualize button ***
                var visualizeButton = document.createElement('button');
                visualizeButton.className = 'intellimenta-color-button action-buttons tooltip';
                visualizeButton.style.transform = 'scale(1)';

                const visualizeButtonIcon = document.createElement('img');
                visualizeButtonIcon.src = visualizeButtonIconURL;

                const visualizeButtonTooltip = document.createElement('span');
                visualizeButtonTooltip.className = 'tooltiptext';
                visualizeButtonTooltip.style.width = 'auto';
                visualizeButtonTooltip.style.whiteSpace = 'nowrap';
                visualizeButtonTooltip.textContent = get_translation('Visualize', language);

                visualizeButton.onclick = function() {
                    visualizeButtonTooltip.style.display = 'none';
                    let visualizeDiv = document.getElementById('visualizeDiv');
                    if (visualizeDiv) {
                        visualizeDiv.remove();
                    } else {
                        visualizeDiv = document.createElement('div');
                        visualizeDiv.id = 'visualizeDiv';
                        visualizeDiv.style.cssText = `
                            display: flex;
                            align-items: flex-start;
                            flex-direction: column;
                            position: absolute;
                            width: 140px;
                            bottom: 100%;
                            left: 0;
                            margin-bottom: 5px;
                            border: 1px solid #ccc;
                            padding: 10px;
                            background-color: #644883;
                            box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.1);
                            border-radius: 5px;
                        `;

                        const dropdown = document.createElement('select');
                        dropdown.style.width = '100%';
                        dropdown.style.marginBottom = '10px';

                        const option1 = document.createElement('option');
                        option1.value = 'Line Chart';
                        option1.textContent = 'Line Chart';

                        const option2 = document.createElement('option');
                        option2.value = 'Bar Chart';
                        option2.textContent = 'Bar Chart';

                        const option3 = document.createElement('option');
                        option3.value = 'Pie Chart';
                        option3.textContent = 'Pie Chart';

                        const option4 = document.createElement('option');
                        option4.value = 'Row Chart';
                        option4.textContent = 'Row Chart';

                        const option5 = document.createElement('option');
                        option5.value = 'Area Chart';
                        option5.textContent = 'Area Chart';

                        const option6 = document.createElement('option');
                        option6.value = 'Combo Chart';
                        option6.textContent = 'Combo Chart';

                        const option7 = document.createElement('option');
                        option7.value = 'Funnel Chart';
                        option7.textContent = 'Funnel Chart';

                        const option8 = document.createElement('option');
                        option8.value = 'Waterfall Chart';
                        option8.textContent = 'Waterfall Chart';

                        const option9 = document.createElement('option');
                        option9.value = 'Scatterplot';
                        option9.textContent = 'Scatterplot';

                        const option10 = document.createElement('option');
                        option10.value = 'Progressbar';
                        option10.textContent = 'Progressbar';

                        const option11 = document.createElement('option');
                        option11.value = 'Trend Visualization';
                        option11.textContent = 'Trend Visualization';

                        const option12 = document.createElement('option');
                        option12.value = 'Map Visualization';
                        option12.textContent = 'Map Visualization';

                        const option13 = document.createElement('option');
                        option13.value = 'Number Visualization';
                        option13.textContent = 'Number Visualization';

                        const option14 = document.createElement('option');
                        option14.value = 'Gauge Visualization';
                        option14.textContent = 'Gauge Visualization';

                        dropdown.appendChild(option1);
                        dropdown.appendChild(option2);
                        dropdown.appendChild(option5);
                        dropdown.appendChild(option3);
                        dropdown.appendChild(option4);
                        dropdown.appendChild(option6);
                        dropdown.appendChild(option8);
                        dropdown.appendChild(option9);
                        dropdown.appendChild(option10);
                        dropdown.appendChild(option11);
                        dropdown.appendChild(option12);
                        dropdown.appendChild(option13);
                        dropdown.appendChild(option14);
                        dropdown.appendChild(option7);                            
                        
                        const submitButton = document.createElement('button');
                        submitButton.textContent = 'Submit';
                        submitButton.style.display = 'block';
                        submitButton.style.margin = 'auto';
                        submitButton.style.cursor = 'pointer';
                        submitButton.onclick = async function(event) {
                            
                            event.stopPropagation();
                            const selectedOption = dropdown.value;                            
                            
                            visualizeDiv.remove();
                            
                            addToConversation(selectedOption);
                            statusUpdatesDiv.innerHTML = '';
                            loadingOverlay.style.display = 'block';
                            var botResponse_ = await askBot(selectedOption);
                            loadingOverlay.style.display = 'none';
                            await handleBotResponse(botResponse_.main_response, false, botResponse_.viz_type, botResponse_.is_single_color);

                            // Scroll to the bottom
                            await scrollToBottom();
                        };

                        visualizeDiv.appendChild(dropdown);
                        visualizeDiv.appendChild(submitButton);

                        visualizeDiv.onclick = function(event) {
                            event.stopPropagation();
                        };

                        visualizeButton.appendChild(visualizeDiv);
                    }
                };

                // Close visualizeDiv when clicking outside
                window.addEventListener('click', function(event) {
                    const visualizeDiv = document.getElementById('visualizeDiv');
                    if (visualizeDiv && !event.target.closest('#visualizeDiv') && !event.target.closest('.intellimenta-color-button')) {
                        visualizeDiv.remove();
                        visualizeButtonTooltip.style.display = 'block';  // enable tooltip again
                    }
                });

                visualizeButton.appendChild(visualizeButtonIcon);
                visualizeButton.appendChild(visualizeButtonTooltip);


                // *** share button ***
                const shareButton = document.createElement('button');
                shareButton.className = 'intellimenta-color-button action-buttons tooltip';

                const shareButtonIcon = document.createElement('img');
                shareButtonIcon.src = shareButtonIconURL;
                
                const shareButtonTooltip = document.createElement('span');
                shareButtonTooltip.className = 'tooltiptext';
                shareButtonTooltip.style.width = 'auto';
                shareButtonTooltip.style.whiteSpace = 'nowrap';
                shareButtonTooltip.textContent = get_translation('Share', language);

                shareButton.onclick = async function() {
                    await askBotPipeline('generate public link');
                };

                shareButton.appendChild(shareButtonIcon);
                shareButton.appendChild(shareButtonTooltip);


                // *** add-to-dashboard button ***
                const addToDashboardButton = document.createElement('button');
                addToDashboardButton.className = 'intellimenta-color-button action-buttons tooltip';
                addToDashboardButton.style.transform = 'scale(1)';

                const addToDashboardButtonIcon = document.createElement('img');
                addToDashboardButtonIcon.src = addToDashboardButtonIconURL;

                const addToDashboardButtonTooltip = document.createElement('span');
                addToDashboardButtonTooltip.className = 'tooltiptext';
                addToDashboardButtonTooltip.style.width = 'auto';
                addToDashboardButtonTooltip.style.whiteSpace = 'nowrap';
                addToDashboardButtonTooltip.textContent = get_translation('Add to Personal Dashboard', language);

                addToDashboardButton.onclick = function() {
                    addToDashboardButtonTooltip.style.display = 'none';
                    let addToDashboardDiv = document.getElementById('addToDashboardDiv');
                    if (addToDashboardDiv) {
                        addToDashboardDiv.remove();
                    } else {
                        addToDashboardDiv = document.createElement('div');
                        addToDashboardDiv.id = 'addToDashboardDiv';
                        addToDashboardDiv.style.cssText = `
                            display: flex;
                            align-items: flex-start;
                            flex-direction: column;
                            position: absolute;
                            width: 200px;
                            bottom: 100%;
                            left: 0;
                            margin-bottom: 5px;
                            border: 1px solid #ccc;
                            padding: 10px;
                            background-color: #644883;
                            box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.1);
                            border-radius: 5px;
                        `;

                        // // 'Create Tab' button
                        // const createTabButton = document.createElement('button');
                        // createTabButton.textContent = 'Create a Tab in Dashboard';
                        // createTabButton.style.margin = 'auto';
                        // createTabButton.style.cursor = 'pointer';

                        // createTabButton.onclick = async function(event) {
                        //     event.stopPropagation();
                        //     addToDashboardDiv.remove();
                            
                        //     await askBotPipeline('create a tab in dashboard');
                        // };
                        
                        // //horizontal line
                        // const hr = document.createElement('hr');
                        // hr.style.width = '95%';
                        // hr.style.margin = '10px auto';

                        // tab number
                        const dropdown = document.createElement('select');
                        dropdown.style.margin = 'auto';
                        dropdown.style.marginLeft = '-65px';

                        for (let i = 1; i <= 10; i++) {
                            const option = document.createElement('option');
                            option.value = i;
                            option.textContent = `Tab ${i}`;
                            dropdown.appendChild(option);
                        }

                        const submitButton = document.createElement('button');
                        submitButton.textContent = 'Add to Dashboard in';
                        submitButton.style.display = 'block';
                        submitButton.style.margin = 'auto';
                        submitButton.style.cursor = 'pointer';
                        submitButton.style.width = '198px';
                        submitButton.style.textAlign = 'left';
                        submitButton.onclick = async function(event) {
                            event.stopPropagation();
                            const tabNumber = dropdown.value || '1';
                            const mapping = {'1':'first', '2':'second', '3':'third', '4':'fourth', '5':'fifth', '6':'sixth', '7':'seventh', '8':'eighth', '9':'ninth', '10':'tenth'}
                            const command = tabNumber == '1' ? 'add to dashboard' : `add to ${mapping[tabNumber]} tab of dashboard `;
                            addToDashboardDiv.remove();
                            
                            await askBotPipeline(command);
                        };
                        
                        const tabNumberAndAddToDashboardDiv = document.createElement('div');
                        tabNumberAndAddToDashboardDiv.style.display = 'flex';
                        tabNumberAndAddToDashboardDiv.appendChild(submitButton);
                        tabNumberAndAddToDashboardDiv.appendChild(dropdown);

                        //addToDashboardDiv.appendChild(createTabButton);
                        //addToDashboardDiv.appendChild(hr);
                        addToDashboardDiv.appendChild(tabNumberAndAddToDashboardDiv);

                        addToDashboardDiv.onclick = function(event) {
                            event.stopPropagation();
                        };
                    }

                    addToDashboardButton.appendChild(addToDashboardDiv);
                };

                // Close addToDashboardDiv when clicking outside
                window.addEventListener('click', function(event) {
                    const addToDashboardDiv = document.getElementById('addToDashboardDiv');
                    if (addToDashboardDiv && !event.target.closest('#addToDashboardDiv') && !event.target.closest('.intellimenta-color-button')) {
                        addToDashboardDiv.remove();
                        addToDashboardButtonTooltip.style.display = 'block';
                    }
                });

                addToDashboardButton.appendChild(addToDashboardButtonIcon);
                addToDashboardButton.appendChild(addToDashboardButtonTooltip);
                

                // *** add to Text-to-SQL translation tests ***
                const addToResponseQualityTestsButton = document.createElement('button');
                if (isAdmin === true) {
                    addToResponseQualityTestsButton.className = 'intellimenta-color-button action-buttons tooltip';

                    const addToResponseQualityTestsButtonIcon = document.createElement('img');
                    addToResponseQualityTestsButtonIcon.src = addToResponseQualityTestsButtonIconURL;
                    
                    const addToResponseQualityTestsButtonTooltip = document.createElement('span');
                    addToResponseQualityTestsButtonTooltip.className = 'tooltiptext';
                    addToResponseQualityTestsButtonTooltip.style.width = '150px';
                    addToResponseQualityTestsButtonTooltip.textContent = get_translation('Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests', language);;

                    addToResponseQualityTestsButton.onclick = async function() {
                        await askBotPipeline('add to Text-to-SQL translation tests');
                    };

                    addToResponseQualityTestsButton.appendChild(addToResponseQualityTestsButtonIcon);
                    addToResponseQualityTestsButton.appendChild(addToResponseQualityTestsButtonTooltip);
                }
                
                // *** feedback button ***
                const feedbackButton = document.createElement('button');
                feedbackButton.className = 'intellimenta-color-button action-buttons tooltip';
                
                feedbackButton.onclick = async function() {
                    await markAsBadResponse();
                    feedbackButton.style.backgroundColor = '#8451bc';
                };

                const feedbackButtonIcon = document.createElement('img');
                feedbackButtonIcon.src = feedbackButtonIconURL;
                feedbackButtonIcon.style.width = '21px';

                const feedbackButtonTooltip = document.createElement('span');
                feedbackButtonTooltip.className = 'tooltiptext';
                feedbackButtonTooltip.style.whiteSpace = 'nowrap';
                feedbackButtonTooltip.textContent = get_translation('Mark as Bad Response', language);

                feedbackButton.appendChild(feedbackButtonIcon);
                feedbackButton.appendChild(feedbackButtonTooltip);
                

                // container for the action buttons
                var actionButtonsContainer = document.createElement('div');
                actionButtonsContainer.id = 'actionButtonsContainer';
                actionButtonsContainer.style.display = 'flex';
                actionButtonsContainer.style.justifyContent = 'flex-start';
                actionButtonsContainer.style.margin = '0px 5px';

                actionButtonsContainer.appendChild(feedbackButton);
                actionButtonsContainer.appendChild(addToDashboardButton);
                if (isAdmin === true) {
                    actionButtonsContainer.appendChild(addToResponseQualityTestsButton);
                }
                actionButtonsContainer.appendChild(shareButton);
                actionButtonsContainer.appendChild(visualizeButton);
                if ( ['bar', 'line', 'row', 'combo', 'area', 'scatter', 'waterfall', 'funnel'].includes(vizType) ) {
                    actionButtonsContainer.appendChild(modifyButton);
                }

                conversationDiv.appendChild(actionButtonsContainer);
            }
            
        } else if (botResponse.startsWith('{')) {
            
            // remove previous action buttons 
            var actionButtonsContainer = document.getElementById('actionButtonsContainer');
            if (actionButtonsContainer) {
                actionButtonsContainer.remove();
            }
            
            await makeViz(botResponse);
            
            if (isAnalyticalMode && isSessionPage == false) {
                // *** feedback button ***
                const feedbackButton = document.createElement('button');
                feedbackButton.className = 'intellimenta-color-button action-buttons tooltip';
                
                feedbackButton.onclick = async function() {
                    await markAsBadResponse();
                    feedbackButton.style.backgroundColor = '#8451bc';
                };

                const feedbackButtonIcon = document.createElement('img');
                feedbackButtonIcon.src = feedbackButtonIconURL;
                feedbackButtonIcon.style.width = '21px';

                const feedbackButtonTooltip = document.createElement('span');
                feedbackButtonTooltip.className = 'tooltiptext';
                feedbackButtonTooltip.style.whiteSpace = 'nowrap';
                feedbackButtonTooltip.textContent = get_translation('Mark as Bad Response', language);

                feedbackButton.appendChild(feedbackButtonIcon);
                feedbackButton.appendChild(feedbackButtonTooltip);

                // container for the action buttons
                var actionButtonsContainer = document.createElement('div');
                actionButtonsContainer.id = 'actionButtonsContainer';
                actionButtonsContainer.style.display = 'flex';
                actionButtonsContainer.style.justifyContent = 'flex-start';
                actionButtonsContainer.style.margin = '0px 5px';

                actionButtonsContainer.appendChild(feedbackButton);

                conversationDiv.appendChild(actionButtonsContainer);
            }
            
        } else {
            // If it's not a link, display it as a regular message
            const botResponseDiv = document.createElement('div');
            if ( language === 'ar-SA' || language === 'fa-IR' || language === 'he-IL' ) {
                botResponseDiv.className = 'bot-response-rtl';
            } else {
                botResponseDiv.className = 'bot-response-ltr';
            }
            botResponseDiv.classList.add('bot-response');
            
            if (botResponse.includes('</table>')) {  // when tables are rendered, it moves all the <br> tags (because we replace \n with <br>) to the top of table
                                                     // so we exclude replacing \n with <br> for the part of the response that contains the table
                // Split around the <table>...</table> block (case-insensitive)
                const parts = botResponse.split(/(<table[\s\S]*?<\/table>)/i);
                for (let i = 0; i < parts.length; i++) {
                    if (!parts[i].toLowerCase().startsWith('<table')) {
                        parts[i] = parts[i]
                            .replace(/\n/g, '<br>')
                            .replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');
                    }
                }
                formattedResponse = parts.join('');
            } else {
                formattedResponse = botResponse
                    .replace(/\r/g, '')   // remove carriage returns
                    .replace(/\n/g, '<br>')
                    .replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');
            }
            botResponseDiv.innerHTML = formattedResponse;
            conversationDiv.appendChild(botResponseDiv);

            if (isAnalyticalMode && isSessionPage == false) {
                await addActionButtonsAnalyticalMode(botResponseDiv);
            }
        }
    }
}


async function addActionButtonsAnalyticalMode(botResponseDiv) {
    // remove previous action buttons 
    var actionButtonsContainer = document.getElementById('actionButtonsContainer');
    if (actionButtonsContainer) {
        actionButtonsContainer.remove();
    }
    
    // *** feedback button ***
    const feedbackButton = document.createElement('button');
    feedbackButton.className = 'intellimenta-color-button action-buttons tooltip';
    
    feedbackButton.onclick = async function() {
        await markAsBadResponse();
        feedbackButton.style.backgroundColor = '#8451bc';
    };

    const feedbackButtonIcon = document.createElement('img');
    feedbackButtonIcon.src = feedbackButtonIconURL;
    feedbackButtonIcon.style.width = '21px';

    const feedbackButtonTooltip = document.createElement('span');
    feedbackButtonTooltip.className = 'tooltiptext';
    feedbackButtonTooltip.style.whiteSpace = 'nowrap';
    feedbackButtonTooltip.textContent = get_translation('Mark as Bad Response', language);

    feedbackButton.appendChild(feedbackButtonIcon);
    feedbackButton.appendChild(feedbackButtonTooltip);


    // *** copy button ***
    const copyButton = document.createElement('button');
    copyButton.className = 'intellimenta-color-button action-buttons tooltip';
    
    copyButton.onclick = async function() {
        //  copy content of botResponseDiv to clipboard
        const textToCopy = botResponseDiv.innerText || botResponseDiv.textContent;
        try {
            await navigator.clipboard.writeText(textToCopy);
            showNotification(get_translation('Copied to clipboard.', language), true);
        } catch (err) {
            console.error('Failed to copy: ', err);
            showNotification(get_translation('Failed to copy to clipboard', language), false);
        }
    };

    const copyButtonIcon = document.createElement('img');
    copyButtonIcon.src = copyButtonIconURL;
    copyButtonIcon.style.width = '21px';

    const copyButtonTooltip = document.createElement('span');
    copyButtonTooltip.className = 'tooltiptext';
    copyButtonTooltip.style.whiteSpace = 'nowrap';
    copyButtonTooltip.textContent = get_translation('Copy', language);

    copyButton.appendChild(copyButtonIcon);
    copyButton.appendChild(copyButtonTooltip);

    
    // *** pdf button ***
    const pdfButton = document.createElement('button');
    pdfButton.className = 'intellimenta-color-button action-buttons tooltip';
    
    pdfButton.onclick = async function() {
        // get the text of the last div with class 'user-message'
        const userMessages = document.querySelectorAll('.user-message');
        const userMessageDiv = userMessages.length ? userMessages[userMessages.length - 1] : null;
        const userMessage = userMessageDiv ? userMessageDiv.innerText : '';
        const pdfContent = `<p class="user-message">User Message: <b>${userMessage}</b></p>
                            Bot Response:<br>
                            ${botResponseDiv.innerHTML}`;

        await downloadPDF(pdfContent, 'Chatbot Response');
    };

    const pdfButtonIcon = document.createElement('img');
    pdfButtonIcon.src = pdfButtonIconURL;
    pdfButtonIcon.style.width = '21px';

    const pdfButtonTooltip = document.createElement('span');
    pdfButtonTooltip.className = 'tooltiptext';
    pdfButtonTooltip.style.whiteSpace = 'nowrap';
    pdfButtonTooltip.textContent = get_translation('Download as PDF', language);

    pdfButton.appendChild(pdfButtonIcon);
    pdfButton.appendChild(pdfButtonTooltip);


    // container for the action buttons
    var actionButtonsContainer = document.createElement('div');
    actionButtonsContainer.id = 'actionButtonsContainer';
    actionButtonsContainer.style.display = 'flex';
    actionButtonsContainer.style.justifyContent = 'flex-start';
    actionButtonsContainer.style.margin = '0px 5px';

    actionButtonsContainer.appendChild(copyButton);
    actionButtonsContainer.appendChild(pdfButton);
    actionButtonsContainer.appendChild(feedbackButton);

    conversationDiv.appendChild(actionButtonsContainer);
}


const translations = {
    "en-US": {
        "Add to Personal Dashboard": "Add to Personal Dashboard",
        "Visualize": "Visualize",
        "Share": "Share",
        "Modify": "Modify",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests",
        "Remove X-Axis Label": "Remove X-Axis Label",
        "Remove Y-Axis Label": "Remove Y-Axis Label",
        "Hide Values On Chart": "Hide Values On Chart",
        "Rotate Labels 45°": "Rotate Labels 45°",
        "Hi! How can I help?": "Hi! How can I help?",
        "Mark as Bad Response": "Mark as Bad Response",
        "Copied to clipboard.": "Copied to clipboard.",
        "Failed to copy to clipboard": "Failed to copy to clipboard",
        "Save failed!": "Save failed!",
        "Saved.": "Saved.",
        "Saved. Refresh the page for the changes to take effect.": "Saved. Refresh the page for the changes to take effect.",
        "Session Deletion Failed!": "Session Deletion Failed!",
        "Tab Creation Failed!": "Tab Creation Failed!",
        "Tab Created Successfully.": "Tab Created Successfully.",
        "Failed to load the dashboard!": "Failed to load the dashboard!",
        "Failed to generate PDF": "Failed to generate PDF",
        "Sync completed.": "Sync completed.",
        "Refresh completed.": "Refresh completed.",
    },
    "fr-CA": {
        "Add to Personal Dashboard": "Ajouter au tableau de bord personnel",
        "Visualize": "Visualiser",
        "Share": "Partager",
        "Modify": "Modifier",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Ajouter la requête et le SQL généré comme nouveau cas de test aux tests de traduction Text-to-SQL",
        "Remove X-Axis Label": "Supprimer l'étiquette de l'axe des X",
        "Remove Y-Axis Label": "Supprimer l'étiquette de l'axe des Y",
        "Hide Values On Chart": "Masquer les valeurs sur le graphique",
        "Rotate Labels 45°": "Faire pivoter les étiquettes de 45°",
        "Hi! How can I help?": "Salut! Comment puis-je vous aider?",
        "Mark as Bad Response": "Marquer comme mauvaise réponse",
        "Copied to clipboard.": "Copié dans le presse-papiers.",
        "Failed to copy to clipboard": "Échec de la copie dans le presse-papiers",
        "Save failed!": "Échec de l'enregistrement!",
        "Saved.": "Enregistré.",
        "Saved. Refresh the page for the changes to take effect.": "Enregistré. Actualisez la page pour que les modifications prennent effet.",
        "Session Deletion Failed!": "Échec de la suppression de la session!",
        "Tab Creation Failed!": "Échec de la création de l'onglet!",
        "Tab Created Successfully.": "Onglet créé avec succès.",
        "Failed to load the dashboard!": "Échec du chargement du tableau de bord!",
        "Failed to generate PDF": "Échec de la génération du PDF",
        "Sync completed.": "Synchronisation terminée.",
        "Refresh completed.": "Rafraîchissement terminé.",
    },
    "fr-FR": {
        "Add to Personal Dashboard": "Ajouter au tableau de bord personnel",
        "Visualize": "Visualiser",
        "Share": "Partager",
        "Modify": "Modifier",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Ajouter la requête et le SQL généré comme nouveau cas de test aux tests de traduction Text-to-SQL",
        "Remove X-Axis Label": "Supprimer l'étiquette de l'axe des X",
        "Remove Y-Axis Label": "Supprimer l'étiquette de l'axe des Y",
        "Hide Values On Chart": "Masquer les valeurs sur le graphique",
        "Rotate Labels 45°": "Faire pivoter les étiquettes de 45°",
        "Hi! How can I help?": "Salut! Comment puis-je vous aider?",
        "Mark as Bad Response": "Marquer comme mauvaise réponse",
        "Copied to clipboard.": "Copié dans le presse-papiers.",
        "Failed to copy to clipboard": "Échec de la copie dans le presse-papiers",
        "Save failed!": "Échec de l'enregistrement!",
        "Saved.": "Enregistré.",
        "Saved. Refresh the page for the changes to take effect.": "Enregistré. Actualisez la page pour que les modifications prennent effet.",
        "Session Deletion Failed!": "Échec de la suppression de la session!",
        "Tab Creation Failed!": "Échec de la création de l'onglet!",
        "Tab Created Successfully.": "Onglet créé avec succès.",
        "Failed to load the dashboard!": "Échec du chargement du tableau de bord!",
        "Failed to generate PDF": "Échec de la génération du PDF",
        "Sync completed.": "Synchronisation terminée.",
        "Refresh completed.": "Rafraîchissement terminé.",
    },
    "de-DE": {
        "Add to Personal Dashboard": "Zum persönlichen Dashboard hinzufügen",
        "Visualize": "Visualisieren",
        "Share": "Teilen",
        "Modify": "Bearbeiten",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Füge die Abfrage und das generierte SQL als neuen Testfall zu den Text-zu-SQL-Übersetzungstests hinzu",
        "Remove X-Axis Label": "X-Achsen-Beschriftung entfernen",
        "Remove Y-Axis Label": "Y-Achsen-Beschriftung entfernen",
        "Hide Values On Chart": "Werte im Diagramm ausblenden",
        "Rotate Labels 45°": "Beschriftungen um 45° drehen",
        "Hi! How can I help?": "Hallo! Wie kann ich helfen?",
        "Mark as Bad Response": "Als schlechte Antwort markieren",
        "Copied to clipboard.": "In die Zwischenablage kopiert.",
        "Failed to copy to clipboard": "Kopieren in die Zwischenablage fehlgeschlagen",
        "Save failed!": "Speichern fehlgeschlagen!",
        "Saved.": "Gespeichert.",
        "Saved. Refresh the page for the changes to take effect.": "Gespeichert. Aktualisieren Sie die Seite, damit die Änderungen wirksam werden.",
        "Session Deletion Failed!": "Sitzungslöschung fehlgeschlagen!",
        "Tab Creation Failed!": "Tab-Erstellung fehlgeschlagen!",
        "Tab Created Successfully.": "Tab erfolgreich erstellt.",
        "Failed to load the dashboard!": "Dashboard konnte nicht geladen werden!",
        "Failed to generate PDF": "PDF konnte nicht erstellt werden",
        "Sync completed.": "Synchronisierung abgeschlossen.",
        "Refresh completed.": "Aktualisierung abgeschlossen.",
    },
    "hi-IN": {
        "Add to Personal Dashboard": "व्यक्तिगत डैशबोर्ड में जोड़ें",
        "Visualize": "दृश्य बनाएं",
        "Share": "साझा करें",
        "Modify": "संशोधित करें",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Text-to-SQL अनुवाद परीक्षणों में क्वेरी और उत्पन्न SQL को एक नए परीक्षण मामले के रूप में जोड़ें",
        "Remove X-Axis Label": "X-अक्ष लेबल हटाएं",
        "Remove Y-Axis Label": "Y-अक्ष लेबल हटाएं",
        "Hide Values On Chart": "चार्ट पर मान छिपाएं",
        "Rotate Labels 45°": "लेबल को 45° घुमाएं",
        "Hi! How can I help?": "नमस्ते! मैं कैसे मदद कर सकता हूँ?",
        "Mark as Bad Response": "बुरी प्रतिक्रिया के रूप में चिह्नित करें",
        "Copied to clipboard.": "क्लिपबोर्ड पर कॉपी किया गया।",
        "Failed to copy to clipboard": "क्लिपबोर्ड पर कॉपी करने में विफल",
        "Save failed!": "सहेजना विफल रहा!",
        "Saved.": "सहेजा गया।",
        "Saved. Refresh the page for the changes to take effect.": "सहेजा गया। परिवर्तनों के लिए पृष्ठ को रीफ्रेश करें।",
        "Session Deletion Failed!": "सत्र हटाना विफल रहा!",
        "Tab Creation Failed!": "टैब बनाना विफल रहा!",
        "Tab Created Successfully.": "टैब सफलतापूर्वक बनाया गया।",
        "Failed to load the dashboard!": "डैशबोर्ड लोड करने में विफल!",
        "Failed to generate PDF": "PDF बनाने में विफल",
        "Sync completed.": "सिंक पूरा हुआ।",
        "Refresh completed.": "रीफ्रेश पूरा हुआ।",
    },
    "ja-JP": {
        "Add to Personal Dashboard": "個人用ダッシュボードに追加",
        "Visualize": "可視化",
        "Share": "共有",
        "Modify": "修正",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "クエリと生成されたSQLをText-to-SQL翻訳テストの新しいテストケースとして追加する",
        "Remove X-Axis Label": "X軸ラベルを削除",
        "Remove Y-Axis Label": "Y軸ラベルを削除",
        "Hide Values On Chart": "チャート上の値を非表示",
        "Rotate Labels 45°": "ラベルを45°回転",
        "Hi! How can I help?": "こんにちは！どのようにお手伝いできますか？",
        "Mark as Bad Response": "悪い応答としてマーク",
        "Copied to clipboard.": "クリップボードにコピーしました。",
        "Failed to copy to clipboard": "クリップボードへのコピーに失敗しました",
        "Save failed!": "保存に失敗しました！",
        "Saved.": "保存しました。",
        "Saved. Refresh the page for the changes to take effect.": "保存しました。変更を反映するにはページを更新してください。",
        "Session Deletion Failed!": "セッションの削除に失敗しました！",
        "Tab Creation Failed!": "タブの作成に失敗しました！",
        "Tab Created Successfully.": "タブが正常に作成されました。",
        "Failed to load the dashboard!": "ダッシュボードの読み込みに失敗しました！",
        "Failed to generate PDF": "PDFの生成に失敗しました",
        "Sync completed.": "同期が完了しました。",
        "Refresh completed.": "更新が完了しました。",
    },
    "pt-BR": {
        "Add to Personal Dashboard": "Adicionar ao painel pessoal",
        "Visualize": "Visualizar",
        "Share": "Compartilhar",
        "Modify": "Modificar",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Adicionar a consulta e o SQL gerado como um novo caso de teste nos testes de tradução Text-to-SQL",
        "Remove X-Axis Label": "Remover rótulo do eixo X",
        "Remove Y-Axis Label": "Remover rótulo do eixo Y",
        "Hide Values On Chart": "Ocultar valores no gráfico",
        "Rotate Labels 45°": "Girar rótulos 45°",
        "Hi! How can I help?": "Oi! Como posso ajudar?",
        "Mark as Bad Response": "Marcar como resposta ruim",
        "Copied to clipboard.": "Copiado para a área de transferência.",
        "Failed to copy to clipboard": "Falha ao copiar para a área de transferência",
        "Save failed!": "Falha ao salvar!",
        "Saved.": "Salvo.",
        "Saved. Refresh the page for the changes to take effect.": "Salvo. Atualize a página para que as alterações tenham efeito.",
        "Session Deletion Failed!": "Falha ao excluir a sessão!",
        "Tab Creation Failed!": "Falha ao criar a guia!",
        "Tab Created Successfully.": "Guia criada com sucesso.",
        "Failed to load the dashboard!": "Falha ao carregar o painel!",
        "Failed to generate PDF": "Falha ao gerar PDF",
        "Sync completed.": "Sincronização concluída.",
        "Refresh completed.": "Atualização concluída.",
    },
    "ru-RU": {
        "Add to Personal Dashboard": "Добавить на персональную панель",
        "Visualize": "Визуализировать",
        "Share": "Поделиться",
        "Modify": "Изменить",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Добавить запрос и сгенерированный SQL как новый тестовый случай в тесты перевода Text-to-SQL",
        "Remove X-Axis Label": "Удалить метку оси X",
        "Remove Y-Axis Label": "Удалить метку оси Y",
        "Hide Values On Chart": "Скрыть значения на графике",
        "Rotate Labels 45°": "Повернуть метки на 45°",
        "Hi! How can I help?": "Привет! Как я могу помочь?",
        "Mark as Bad Response": "Отметить как плохой ответ",
        "Copied to clipboard.": "Скопировано в буфер обмена.",
        "Failed to copy to clipboard": "Не удалось скопировать в буфер обмена",
        "Save failed!": "Ошибка сохранения!",
        "Saved.": "Сохранено.",
        "Saved. Refresh the page for the changes to take effect.": "Сохранено. Обновите страницу, чтобы изменения вступили в силу.",
        "Session Deletion Failed!": "Ошибка удаления сессии!",
        "Tab Creation Failed!": "Ошибка создания вкладки!",
        "Tab Created Successfully.": "Вкладка успешно создана.",
        "Failed to load the dashboard!": "Не удалось загрузить панель!",
        "Failed to generate PDF": "Не удалось создать PDF",
        "Sync completed.": "Синхронизация завершена.",
        "Refresh completed.": "Обновление завершено.",
    },
    "es-ES": {
        "Add to Personal Dashboard": "Agregar al panel personal",
        "Visualize": "Visualizar",
        "Share": "Compartir",
        "Modify": "Modificar",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Agregar la consulta y el SQL generado como un nuevo caso de prueba a las pruebas de traducción Text-to-SQL",
        "Remove X-Axis Label": "Eliminar etiqueta del eje X",
        "Remove Y-Axis Label": "Eliminar etiqueta del eje Y",
        "Hide Values On Chart": "Ocultar valores en el gráfico",
        "Rotate Labels 45°": "Rotar etiquetas 45°",
        "Hi! How can I help?": "¡Hola! ¿Cómo puedo ayudar?",
        "Mark as Bad Response": "Marcar como respuesta mala",
        "Copied to clipboard.": "Copiado al portapapeles.",
        "Failed to copy to clipboard": "Error al copiar al portapapeles",
        "Save failed!": "¡Error al guardar!",
        "Saved.": "Guardado.",
        "Saved. Refresh the page for the changes to take effect.": "Guardado. Actualiza la página para que los cambios tengan efecto.",
        "Session Deletion Failed!": "¡Error al eliminar la sesión!",
        "Tab Creation Failed!": "¡Error al crear la pestaña!",
        "Tab Created Successfully.": "Pestaña creada con éxito.",
        "Failed to load the dashboard!": "¡Error al cargar el panel!",
        "Failed to generate PDF": "Error al generar el PDF",
        "Sync completed.": "Sincronización completada.",
        "Refresh completed.": "Actualización completada.",
    },
    "ar-SA": {
        "Add to Personal Dashboard": "أضف إلى لوحة المعلومات الشخصية",
        "Visualize": "تصور",
        "Share": "مشاركة",
        "Modify": "تعديل",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "أضف الاستعلام وSQL المُولّد كحالة اختبار جديدة لاختبارات ترجمة Text-to-SQL",
        "Remove X-Axis Label": "إزالة تسمية المحور السيني",
        "Remove Y-Axis Label": "إزالة تسمية المحور الصادي",
        "Hide Values On Chart": "إخفاء القيم على الرسم البياني",
        "Rotate Labels 45°": "تدوير الملصقات 45 درجة",
        "Hi! How can I help?": "مرحبًا! كيف يمكنني المساعدة؟",
        "Mark as Bad Response": "وضع علامة على الاستجابة السيئة",
        "Copied to clipboard.": "تم النسخ إلى الحافظة",
        "Failed to copy to clipboard": "فشل النسخ إلى الحافظة",
        "Save failed!": "فشل الحفظ",
        "Saved.": "تم الحفظ",
        "Saved. Refresh the page for the changes to take effect.": "تم الحفظ. يرجى تحديث الصفحة لتطبيق التغييرات",
        "Session Deletion Failed!": "فشل حذف الجلسة",
        "Tab Creation Failed!": "فشل إنشاء التبويب",
        "Tab Created Successfully.": "تم إنشاء التبويب بنجاح",
        "Failed to load the dashboard!": "فشل تحميل لوحة المعلومات",
        "Failed to generate PDF": "فشل إنشاء ملف PDF",
        "Sync completed.": "اكتمل المزامنة.",
        "Refresh completed.": "اكتمل التحديث.",
    },
    "fa-IR": {
        "Add to Personal Dashboard": "افزودن به داشبورد شخصی",
        "Visualize": "رسم نمودار",
        "Share": "به اشتراک گذاری",
        "Modify": "ایجاد تغییرات",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "اضافه کردن سوال و کد تولید شده به عنوان یک مورد آزمایشی جدید به آزمون های ترجمه Text-to-SQL",
        "Remove X-Axis Label": "حذف برچسب محور افقی",
        "Remove Y-Axis Label": "حذف برچسب محور عمودی",
        "Hide Values On Chart": "مخفی کردن مقادیر روی نمودار",
        "Rotate Labels 45°": "چرخش برچسب ها به اندازه 45 درجه",
        "Hi! How can I help?": "سلام! چطور می توانم کمک کنم؟",
        "Mark as Bad Response": "علامت گذاری به عنوان پاسخ بد",
        "Copied to clipboard.": "در کلیپ‌بورد کپی شد",
        "Failed to copy to clipboard": "کپی در کلیپ‌بورد ناموفق بود",
        "Save failed!": "ذخیره‌سازی ناموفق بود",
        "Saved.": "ذخیره شد",
        "Saved. Refresh the page for the changes to take effect.": "ذخیره شد. برای اعمال تغییرات صفحه را ریفرش کنید",
        "Session Deletion Failed!": "حذف ناموفق بود",
        "Tab Creation Failed!": "ایجاد تب ناموفق بود",
        "Tab Created Successfully.": "تب با موفقیت ایجاد شد",
        "Failed to load the dashboard!": "بارگذاری داشبورد ناموفق بود",
        "Failed to generate PDF": "تولید PDF ناموفق بود",
        "Sync completed.": "همگام‌سازی تکمیل شد.",
        "Refresh completed.": "به‌روزرسانی تکمیل شد.",
    },
    "zh-CN": {
        "Add to Personal Dashboard": "添加到个人仪表板",
        "Visualize": "可视化",
        "Share": "分享",
        "Modify": "修改",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "将查询和生成的SQL作为新的测试用例添加到Text-to-SQL翻译测试中",
        "Remove X-Axis Label": "删除X轴标签",
        "Remove Y-Axis Label": "删除Y轴标签",
        "Hide Values On Chart": "隐藏图表上的值",
        "Rotate Labels 45°": "旋转标签45°",
        "Hi! How can I help?": "嗨！我能帮你什么？",
        "Mark as Bad Response": "标记为不良响应",
        "Copied to clipboard.": "已复制到剪贴板。",
        "Failed to copy to clipboard": "复制到剪贴板失败",
        "Save failed!": "保存失败！",
        "Saved.": "已保存。",
        "Saved. Refresh the page for the changes to take effect.": "已保存。请刷新页面以使更改生效。",
        "Session Deletion Failed!": "会话删除失败！",
        "Tab Creation Failed!": "标签创建失败！",
        "Tab Created Successfully.": "标签创建成功。",
        "Failed to load the dashboard!": "仪表板加载失败！",
        "Failed to generate PDF": "生成PDF失败",
        "Sync completed.": "同步完成。",
        "Refresh completed.": "刷新完成。",
    },
    "zh-TW": {
        "Add to Personal Dashboard": "新增至個人儀表板",
        "Visualize": "視覺化",
        "Share": "分享",
        "Modify": "修改",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "將查詢與產生的SQL作為新的測試案例新增至Text-to-SQL翻譯測試中",
        "Remove X-Axis Label": "移除X軸標籤",
        "Remove Y-Axis Label": "移除Y軸標籤",
        "Hide Values On Chart": "隱藏圖表上的值",
        "Rotate Labels 45°": "旋轉標籤45°",
        "Hi! How can I help?": "嗨！我能幫助你什麼？",
        "Mark as Bad Response": "標記為不良回應",
        "Copied to clipboard.": "已複製到剪貼簿。",
        "Failed to copy to clipboard": "複製到剪貼簿失敗",
        "Save failed!": "儲存失敗！",
        "Saved.": "已儲存。",
        "Saved. Refresh the page for the changes to take effect.": "已儲存。請重新整理頁面以套用變更。",
        "Session Deletion Failed!": "刪除會話失敗！",
        "Tab Creation Failed!": "建立分頁失敗！",
        "Tab Created Successfully.": "分頁建立成功。",
        "Failed to load the dashboard!": "儀表板載入失敗！",
        "Failed to generate PDF": "產生PDF失敗",
        "Sync completed.": "同步完成。",
        "Refresh completed.": "重新整理完成。",
    },
    "bn-BD": {
        "Add to Personal Dashboard": "ব্যক্তিগত ড্যাশবোর্ডে যোগ করুন",
        "Visualize": "চিত্রায়িত করুন",
        "Share": "শেয়ার করুন",
        "Modify": "পরিবর্তন করুন",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "Text-to-SQL অনুবাদ পরীক্ষায় একটি নতুন টেস্ট কেস হিসাবে কুয়েরি এবং তৈরি করা SQL যুক্ত করুন",
        "Remove X-Axis Label": "X-অক্ষের লেবেল সরান",
        "Remove Y-Axis Label": "Y-অক্ষের লেবেল সরান",
        "Hide Values On Chart": "চার্টে মান লুকান",
        "Rotate Labels 45°": "লেবেল 45° ঘোরান",
        "Hi! How can I help?": "হ্যালো! আমি কিভাবে সাহায্য করতে পারি?",
        "Mark as Bad Response": "খারাপ প্রতিক্রিয়া হিসাবে চিহ্নিত করুন",
        "Copied to clipboard.": "ক্লিপবোর্ডে কপি হয়েছে।",
        "Failed to copy to clipboard": "ক্লিপবোর্ডে কপি করতে ব্যর্থ",
        "Save failed!": "সংরক্ষণ ব্যর্থ হয়েছে!",
        "Saved.": "সংরক্ষিত হয়েছে।",
        "Saved. Refresh the page for the changes to take effect.": "সংরক্ষিত হয়েছে। পরিবর্তন কার্যকর করতে পৃষ্ঠা রিফ্রেশ করুন।",
        "Session Deletion Failed!": "সেশন মুছে ফেলা ব্যর্থ হয়েছে!",
        "Tab Creation Failed!": "ট্যাব তৈরি ব্যর্থ হয়েছে!",
        "Tab Created Successfully.": "ট্যাব সফলভাবে তৈরি হয়েছে।",
        "Failed to load the dashboard!": "ড্যাশবোর্ড লোড করতে ব্যর্থ!",
        "Failed to generate PDF": "PDF তৈরি ব্যর্থ হয়েছে",
        "Sync completed.": "সিঙ্ক সম্পন্ন হয়েছে।",
        "Refresh completed.": "রিফ্রেশ সম্পন্ন হয়েছে।",
    },
    "he-IL": {
        "Add to Personal Dashboard": "הוסף ללוח מחוונים אישי",
        "Visualize": "ויזואליזציה",
        "Share": "שתף",
        "Modify": "שנה",
        "Add the query and the generated SQL as a new test case to the Text-to-SQL translation tests": "הוסף את השאילתה ואת ה-SQL שנוצרו כמקרה בדיקה חדש לבדיקות תרגום Text-to-SQL",
        "Remove X-Axis Label": "הסר תווית ציר X",
        "Remove Y-Axis Label": "הסר תווית ציר Y",
        "Hide Values On Chart": "הסתר ערכים בגרף",
        "Rotate Labels 45°": "סובב תוויות 45°",
        "Hi! How can I help?": "שלום! איך אני יכול לעזור?",
        "Mark as Bad Response": "סמן כתשובה רעה",
        "Copied to clipboard.": "הועתק ללוח",
        "Failed to copy to clipboard": "ההעתקה ללוח נכשלה",
        "Save failed!": "השמירה נכשלה",
        "Saved.": "נשמר",
        "Saved. Refresh the page for the changes to take effect.": "נשמר. רענן את הדף כדי שהשינויים ייכנסו לתוקף",
        "Session Deletion Failed!": "מחיקת הסשן נכשלה",
        "Tab Creation Failed!": "יצירת הלשונית נכשלה",
        "Tab Created Successfully.": "הלשונית נוצרה בהצלחה",
        "Failed to load the dashboard!": "טעינת לוח המחוונים נכשלה",
        "Failed to generate PDF": "יצירת PDF נכשלה",
        "Sync completed.": "סנכרון הושלם.",
        "Refresh completed.": "רענון הושלם.",
    }
};


function get_translation(key, locale = 'en') {
    return translations[locale]?.[key] || key;
}



async function loadGreetingsAndQuerySuggestions() {
    const greetingDiv = document.createElement('div');
    greetingDiv.id = 'greetingDiv';
    greetingDiv.style.marginTop = '23%';

    const image = document.createElement('img');
    image.src = logoUrl;
    image.alt = 'Logo';
    image.style.width = `${logoMaxWidth* 0.8}px`;
    greetingDiv.appendChild(image);

    const greetingParagraph = document.createElement('p');
    greetingParagraph.id='greeting';
    if ( isFirstTimeUser === true ) {
        greetingParagraph.style.padding = '25px';
        greetingParagraph.style.fontSize = '17px';
        if ( wla ) {
            greetingParagraph.innerHTML = demoUrl 
                ? `For a quick overview of the app, please watch this <a href="${demoUrl}" target="_blank">demo video</a>.` 
                : get_translation('Hi! How can I help?', language);
        } else {
            greetingParagraph.innerHTML = isAnalyticalMode
                ? 'For a quick overview of the app, please watch this <a href="https://www.loom.com/share/d3746a993eca463eb5100ce65104cfb6" target="_blank">demo video</a>.'
                : 'For a quick overview of the app, please watch this <a href="https://www.loom.com/share/d1af20ba05f94be2b9d84c5c26937463" target="_blank">demo video</a>.';
        }
        
    } else {
        greetingParagraph.textContent  = get_translation('Hi! How can I help?', language);
    }
    greetingDiv.appendChild(greetingParagraph);
    
    if (querySuggestions && Object.keys(querySuggestions).length > 0) {
        const querySuggestionsDiv = document.createElement('div');
        querySuggestionsDiv.id = 'query-suggestions';
        
        // get active db id
        const dropdown = document.getElementById('ActiveDatabaseDropdown');
        const selectedOption = dropdown.options[dropdown.selectedIndex];
        const activeDbId = selectedOption.id;

        const maxQueryLength =  (mediaQuery.matches || isChatbotAndDashboards) ? 120 : 60; 
        if (querySuggestions[activeDbId]) {
            querySuggestions[activeDbId].forEach(query => {
                if (query != '') {
                    const button = document.createElement("button");
                    button.className = "query-suggestion";
                    button.textContent = query.length > maxQueryLength ? query.substring(0, maxQueryLength) + '...' : query;
                    if (query.length > 60) {
                        button.title = query;
                    }
                    button.onclick = () => askBotPipeline(query);
                    querySuggestionsDiv.appendChild(button);
                }
            });
        }
        greetingDiv.appendChild(querySuggestionsDiv);
    }

    conversationDiv.appendChild(greetingDiv);
}



function addToConversation(message) {
    const messageDiv = document.createElement('div');
    if (mediaQuery.matches || isChatbotAndDashboards) {
        messageDiv.style.cssText = 'padding: 5px; background-color: #6448833e; font-size: 15px; font-family:"Merriweather", "Rubik", sans-serif';
    } else {
        messageDiv.style.cssText = 'padding: 5px; background-color: #6448833e; font-size: 15px; font-family:"Merriweather", "Rubik", sans-serif; border-radius: 5px; margin: 10px 0px;';
    }
    messageDiv.className = 'user-message';

    if (language === 'ar-SA' || language === 'fa-IR' || language === 'he-IL') {
        messageDiv.style.textAlign = 'right';
        messageDiv.style.direction = 'rtl';
    } else {
        messageDiv.style.textAlign = 'left';
        messageDiv.style.direction = 'ltr';
    }

    // Preserve newlines by replacing them with <br>
    messageDiv.innerHTML = message
        .replace(/\r/g, '')   // remove carriage returns
        .replace(/\n/g, '<br>')
        .replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');

    conversationDiv.appendChild(messageDiv);

    // ask user to start a new chat on high message count
    const messages = conversationDiv.getElementsByClassName('user-message');
    const messageCount = messages.length;
    if (isSession == false && (messageCount == 5 || messageCount == 8 || messageCount == 11)) {
        showNotification(`For best results, open a new chat (click on the plus icon at the top) if your next question is on a different topic.`, true, 4500);
    }
}


async function removeGreetingsAndSuggestionsDivs() {
    // remove suggestions div
    let querySuggestionsDiv = document.getElementById('query-suggestions');
    if (querySuggestionsDiv) {
        querySuggestionsDiv.style.display = 'none';
    }
    
    // remove the greeting div
    const greetingsDiv = document.getElementById('greetingDiv');
    if (greetingsDiv) {
        greetingsDiv.style.display = 'none';
    }
}



async function askBot(userQuery) {
    
    await scrollToBottom();

    let activeDbId = 0;
    let activeDbEngine = '';
    if (userQuery != 'show_personal_dashboard' && userQuery != 'clear_session') {
        // get the id of the selected active database
        const dropdown = document.getElementById("ActiveDatabaseDropdown");
        const selectedIndex = dropdown.selectedIndex;
        if (selectedIndex !== -1) {
            activeDbId = dropdown.options[selectedIndex].id;
            activeDbEngine = dropdown.options[selectedIndex].getAttribute('engine');
        }
        if (activeDbId == 0 && !userQuery.startsWith('?')) {
            if (isAdmin === true) {
                return {'main_response':'You need to first select a database from the dropdown at the top. If no option is available, please first add a database connection in the admin panel.'};
            } else {
                return {'main_response':'You need to first select a database from the dropdown at the top. If no option is available, please ask the admin to add a database connection.'};
            }
        }
    }

    const response = await fetch('/chat', {
        method: 'POST',
        body: new URLSearchParams({ userMessage: userQuery, activeDbId: activeDbId, activeDbEngine:activeDbEngine }),
    });
    const botResponse = await response.json();
    return botResponse;
}



async function askBotPipeline(userMessage) {

    try {
        
        statusUpdatesDiv.innerHTML = '';

        loadingOverlay.style.display = 'block';  // show spinner
        
        await removeGreetingsAndSuggestionsDivs();
        
        // addToCommandHistory(userMessage.trim());  // this is the temp command history accessed by arrow keys. 
        // [removed in favor of shift+enter functionality which requires multiple lines and therefore the up/down arrow keys cannot be used for command history]
        
        document.getElementById('user_message').value = ''; // Clear the input field
        
        addToConversation(userMessage);  // Append user's message to the current conversation

        await scrollToBottom();

        let activeDbId = 0;
        let activeDbEngine = '';
        if (userMessage != 'show_personal_dashboard' && userMessage != 'clear_session') {
            // get the id of the selected active database
            const dropdown = document.getElementById("ActiveDatabaseDropdown");
            const selectedIndex = dropdown.selectedIndex;
            if (selectedIndex !== -1) {
                activeDbId = dropdown.options[selectedIndex].id;
                activeDbEngine = dropdown.options[selectedIndex].getAttribute('engine');
            }
            if (activeDbId == 0 && !userMessage.startsWith('?')) {
                loadingOverlay.style.display = 'none'; 
                if (isAdmin === true) {
                    handleBotResponse('You need to first select a database from the dropdown at the top. If no option is available, please first add a database connection in the admin panel.');
                } else {
                    handleBotResponse('You need to first select a database from the dropdown at the top. If no option is available, please ask the admin to add a database connection.');
                }
                return;
            }
        }

        const data = {
            userMessage: userMessage,
            activeDbId: activeDbId,
            activeDbEngine: activeDbEngine,
        };
        
        socket.send(JSON.stringify(data));  // chat() function will be called by the /ws endpoint
    
    } catch (error) {
        loadingOverlay.style.display = 'none';  
        await handleBotResponse("An error occurred: " + error.stack); // show the error to the user
    }    
}


async function clearConversationHistory(withSpinner=true, clearSessionFromDB=false) { 
    
    if (isSession !== true) {  // if it is not a session page (i.e. it's the regular main page)
        try {
            if (withSpinner) {
                loadingOverlay.style.display = 'block';  // show spinner
            }
            
            // clear frontend
            document.getElementById('conversation').innerHTML = '';

            // clear backend
            if (clearSessionFromDB) {
                const botResponse = await askBot('clear_session');
                await handleBotResponse(botResponse.main_response, true);
            }

            if (withSpinner) {
                loadingOverlay.style.display = 'none';  // hide spinner
            }

        } catch (error) {
            if (withSpinner) {
                loadingOverlay.style.display = 'none';  // hide spinner
            }
            await handleBotResponse("An error occurred: " + error.stack); // show the error to the user
        }
    } else {
        for (const [query, response] of chatSession) {
            addToConversation(query);
            await handleBotResponse(response, false, '', false, true);
        }
    }
};



function togglePopup() {
    if (popup.style.display === "none" || popup.style.display === "") {
        popup.style.display = "block";
    } else {
        popup.style.display = "none";
    }
}


function closePopup() {
    popup.style.display = "none";
}



async function saveUserLevelInstructions() {
    const instructions = document.getElementById('user-level-instructions').value;
    const responseDiv = document.getElementById('responseUserLevelInstructions');
    const spinner = document.getElementById('spinnerSaveUserLevelInstructions');

    try {
        spinner.style.display = 'block'; 

        const response = await fetch('/save-user-level-instructions', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ instructions: instructions })
        });
        const result = await response.text();

        spinner.style.display = 'none';

        responseDiv.textContent = result;
        if (result.startsWith('Error')) { responseDiv.style.color = 'red'; }
        else { responseDiv.style.color = 'green'; }

    } catch (error) {
        spinner.style.display = 'none';
        responseDiv.innerText = `An error occured while saving (${error})`;
        responseDiv.style.color = 'red'; 
    }
}



async function saveUserPreference(preferenceName, preferenceValue) {
    if (preferenceName == 'langPreference') { 
        langPref = preferenceValue;
        const statusUpdatesDiv = document.getElementById('status-updates');
        if ( langPref == 'ar-SA' || langPref == 'fa-IR' || langPref == 'he-IL' ) {
            statusUpdatesDiv.style.textAlign = 'right';
            statusUpdatesDiv.style.direction = 'rtl';
        } else {
            statusUpdatesDiv.style.textAlign = 'left';
            statusUpdatesDiv.style.direction = 'ltr';
        }
    }
    if (preferenceName == 'showInterimResult') { showInterimResultPref = preferenceValue; }
    if (preferenceName == 'cardHeightPreference') { cardHeightPref = preferenceValue; }

    const response = await fetch('/save-user-preference', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ [preferenceName]: preferenceValue })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        if (preferenceName == 'langPreference') {
            showNotification(get_translation("Saved. Refresh the page for the changes to take effect.", preferenceValue), true);
        }
        // also set the height for the currntly displayed iframes
        if (preferenceName == 'cardHeightPreference') {
            const iframes = document.querySelectorAll('#conversation iframe');
            iframes.forEach(iframe => {
                iframe.style.height = preferenceValue;
            });
        }
        
    } 
}


async function saveLLMconfig() {
    const llm_openai_api_key = document.getElementById('llm_openai_api_key').value;
    const responseDiv = document.getElementById('responseLLMconfig');
    const spinner = document.getElementById('spinnerSaveLLMconfig');

    try {
        responseDiv.textContent = '';
        spinner.style.display = 'block'; 

        const response = await fetch('/save-llm-config', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ llm_openai_api_key: llm_openai_api_key })
        });
        const result = await response.text();

        spinner.style.display = 'none';

        responseDiv.textContent = result;
        if (result.startsWith('Error')) { responseDiv.style.color = 'red'; }
        else { responseDiv.style.color = 'green'; }

    } catch (error) {
        spinner.style.display = 'none';
        responseDiv.innerText = `An error occured while saving (${error})`;
        responseDiv.style.color = 'red'; 
    }
}


async function saveDefaultLangForAllUsers(language) {
    const statusUpdatesDiv = document.getElementById('status-updates');
    if ( language == 'ar-SA' || language == 'fa-IR' || language == 'he-IL' ) {
        statusUpdatesDiv.style.textAlign = 'right';
        statusUpdatesDiv.style.direction = 'rtl';
    } else {
        statusUpdatesDiv.style.textAlign = 'left';
        statusUpdatesDiv.style.direction = 'ltr';
    }

    const response = await fetch('/save-default-language-for-all-users', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ default_language: language })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved. Refresh the page for the changes to take effect.", language), true);
    } 
}



async function saveAnalyticalModePreference(value) {
    const response = await fetch('/save-analytical-mode-preference', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ analyticalMode:value })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function savePublicSharingPreference(value) {
    const response = await fetch('/save-public-sharing-preference', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ allowPublicSharing:value })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveAppMode(value) {
    const schemaModeText = document.getElementById('schema-mode-text');
    const analyticalModeText = document.getElementById('analytical-mode-text');
    if (value == true) {
        schemaModeText.style.fontWeight = 'normal';
        analyticalModeText.style.fontWeight = 'bold';
    } else {
        schemaModeText.style.fontWeight = 'bold';
        analyticalModeText.style.fontWeight = 'normal';
    }
    const response = await fetch('/save-app-mode', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ appMode:value })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveSendErrorReportsPreference(value) {
    const response = await fetch('/save-send-error-reports-preference', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ sendErrorReports:value })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveSendAnonymosUsageDataPreference(value) {
    const response = await fetch('/save-send-anonymous-usage-data-preference', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ sendAnonymousUsageData:value })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveDataObfuscation(value, groupName, dbId, tableId, fieldId) {
    const response = await fetch('/save-data-obfuscation', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ toggleChecked:value, groupName:groupName, dbId:dbId, tableId:tableId, fieldId:fieldId })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveUserAccess(email, hasAccess) {
    const response = await fetch('/save-user-access', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({  'email': email, 'hasAccess': hasAccess })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveUserGroupMembersip(email, group, updateType) {
    const response = await fetch('/save-user-group-membership', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({  'email': email, 'group': group, 'update_type': updateType })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveDBdesc(dbId) {

    const spinner = document.getElementById('spinnerMetadataUpdate');
    spinner.style.display = 'block';

    const dbDescription = document.getElementById(`db_${dbId}`).value;

    const response = await fetch('/save-db-description', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({  'dbId': dbId, 'dbDescription': dbDescription })
    });
    spinner.style.display = 'none';
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function saveTablePermission(dbId, tableId, group, updateType) {
    const response = await fetch('/save-table-permission', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ 'dbId':dbId, 'tableId': tableId, 'group': group, 'update_type': updateType })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved. Refresh the page for the changes to take effect.", language), true);
    } 
}



async function saveWhiteLabelingConfig() {
    const clientAppName = document.getElementById('client-app-name').value;
    const clientLogoUrl = document.getElementById('client-logo-url').value;
    const clientFaviconUrl = document.getElementById('client-favicon-url').value;
    const clientLogoMaxWidth = document.getElementById('client-logo-max-width').value;
    const clientDemoUrl = document.getElementById('client-demo-url').value;

    const response = await fetch('/save-white-labeling-config', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ 'clientAppName': clientAppName, 'clientLogoUrl': clientLogoUrl, 'clientLogoMaxWidth': clientLogoMaxWidth, 'clientFaviconUrl': clientFaviconUrl, 'clientDemoUrl': clientDemoUrl })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


function openTab(tabName) {
    
    if (tabName === 'querySuggestions') {
        // clear 'Saved.' message when re-opening the tab
        const response = document.getElementById('responseQuerySuggestionSetup');
        response.textContent = '';
    }

    document.getElementById(tabName).classList.add('active');

    
}


async function openTabNoPreLoad(tabName, endpoint) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch(`get-html-content-for-${endpoint}`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadRunResponseQualityTests(dbId) {
    
    const spinner = document.getElementById('spinnerSaveResponseQualityTests');
    spinner.style.display = 'block';

    const response = await fetch('/get-html-content-for-run-response-quality-tests', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({dbId:dbId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    const tab = document.getElementById('responseQualityTestRunResults');
    tab.innerHTML = result;
    tab.classList.add('active');

    if (!result.includes('There is no test defined.')) {
        // run the tests
        const data = { request_type: 'run_response_quality_tests', db_id: dbId };
        socket.send(JSON.stringify(data));
    }
    else {
        const spinnerTestRunResults = document.getElementById('spinnerTestRunResults');
        spinnerTestRunResults.style.visibility = 'hidden';
    }
}


async function openTabNoPreLoadCustomUserAttributeAssignment(attributeName) {
    
    const spinner = document.getElementById('spinnerAttributeAssignment');
    spinner.style.display = 'block';

    const response = await fetch('/get-html-content-for-custom-user-attributes-assignment', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({attributeName:attributeName})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById('customUserAttributeAssignment');
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreloadDatabaseConnection(dbId) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';

    const response = await fetch('/get-html-content-for-database-connection', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({dbId:dbId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById('db_ref');
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadMetadata(tabName) {
    
    const spinner = document.getElementById('spinnerMetadataUpdate');
    spinner.style.display = 'block';
    
    const nameParts = tabName.split('_');
    const dbId = nameParts[0];
    const tableId = nameParts[1];

    const response = await fetch('/get-html-content-for-table-metadata', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({dbId:dbId, tableId:tableId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    var fieldsDiv = document.getElementById(tabName + '_fields');
    fieldsDiv.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadrlac(tabName) {
    
    const spinner = document.getElementById('spinnerrlac');
    spinner.style.display = 'block';
    
    const nameParts = tabName.split('_');
    const groupName = nameParts[1]; 
    const dbId = nameParts[2];
    const tableId = nameParts[3];

    const response = await fetch('/get-html-content-for-rlac', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({groupName:groupName, dbId:dbId, tableId:tableId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    var fieldsDiv = document.getElementById(tabName + '_fields');
    fieldsDiv.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadDataObfuscation(tabName) {
    
    const spinner = document.getElementById('spinnerDataObfuscation');
    spinner.style.display = 'block';
    
    const nameParts = tabName.split('_');
    const groupName = nameParts[1]; 
    const dbId = nameParts[2];
    const tableId = nameParts[3];

    const response = await fetch('/get-html-content-for-data-obfuscation', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({groupName:groupName, dbId:dbId, tableId:tableId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    var fieldsDiv = document.getElementById(tabName + '_fields');
    fieldsDiv.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadTwoColumnSetup(tabName) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-two-column-setup', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}



async function openTabNoPreLoadPerformanceTuning(tabName) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-performance-tuning', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}



async function openTabNoPreLoadEmailSetup(tabName) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-email-setup', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadErrorLogs(tabName) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-error-logs', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadGroupUserList(tabName) {
    
    const spinner = document.getElementById('spinnerGroups');
    spinner.style.display = 'block';
    
    const nameParts = tabName.split('_');
    const groupName = nameParts[1]; 

    const response = await fetch('/get-html-content-for-groups-user-list', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({groupName:groupName})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    var fieldsDiv = document.getElementById(tabName + '_users');
    fieldsDiv.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadWhiteLabeling(tabName) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';

    const response = await fetch('/get-html-content-for-white-labeling', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadResponseQualityTests(dbId) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-response-quality-tests', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({dbId:dbId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById('responseQualityTests');
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadQuerySuggestions(dbId) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-query-suggestions', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({dbId:dbId})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById('querySuggestions');
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function openTabNoPreLoadManageAccessToApp(tabName) {
    
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';
    
    const response = await fetch('/get-html-content-for-manage-access-to-app', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    spinner.style.display = 'none';
    const result = await response.text();
    
    var tab = document.getElementById(tabName);
    tab.innerHTML = result;
    tab.classList.add('active');
}


async function toggleAllowedEmailDomainsDiv() {
    const allowedEmailDomainsDiv = document.getElementById('allowedEmailDomainsDiv');
    const allowedEmailDomainsCheckbox = document.getElementById('toggle-button-email-domain-whitelist');
    
    if (allowedEmailDomainsCheckbox.checked) {
        allowedEmailDomainsDiv.style.display = 'block';
    } else {
        allowedEmailDomainsDiv.style.display = 'none';
    }
}


async function toggleUseUploadedFilesAsDataSource(isChecked) {

    const response = await fetch('/save-use-uploaded-files-as-data-source', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({useAsDataSource:isChecked})
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function toggleRefreshCardsBtn() {
    const section = document.getElementById('refresh_cards_data_in_db_btn');
    const toggleBtn = document.getElementById('toggle-button-use-dashboard-cards-data-as-data-source');
    
    if (toggleBtn.checked) {
        section.style.display = 'flex';
    } else {
        section.style.display = 'none';
    }

    // also save the toggle preference
    const useDashboardCardsAsDataSource = document.getElementById('toggle-button-use-dashboard-cards-data-as-data-source').checked;
    const response = await fetch('/save-bi-integration-config', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            useDashboardCardsAsDataSource:useDashboardCardsAsDataSource,
        })
    });
    
    const result = await response.text();
    if (result.includes('Error')) {
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    }
    else {
        showNotification(get_translation("Saved.", language), true);
    }
}


async function refreshDashboardsCardsInDb() {
    
    const spinner = document.getElementById('spinnerRefreshCardData');
    const resultDiv = document.getElementById('responseRefreshCardData');

    spinner.style.display = 'block';

    const response = await fetch('/refresh-dashboards-cards-in-db', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({})
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        spinner.style.display = 'none';
        resultDiv.style.color = 'red';
        resultDiv.innerText = 'Sorry, there was an error while refreshing the cards data. You can see the error in the browser console and in the erorr_logs table.';
        console.log(result);
    } else {
        spinner.style.display = 'none';
        // green if result == 'Saved.' else blue
        resultDiv.style.color = result === 'Saved.' ? 'green' : 'blue';
        resultDiv.innerText = result;
    } 
}



async function saveAccessToAppConfig() {
    
    const toggleBtnManualAccess = document.getElementById('toggle-button-manual-access-grant').checked;
    const toggleBtnEmailDomainWhitelist = document.getElementById('toggle-button-email-domain-whitelist').checked;
    const textAreaallowedEmailDomains = document.getElementById('allowedEmailDomains').value;

    const response = await fetch('/save-access-to-app-config', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ 
            'toggleBtnManualAccess': toggleBtnManualAccess, 
            'toggleBtnEmailDomainWhitelist': toggleBtnEmailDomainWhitelist, 
            'textAreaallowedEmailDomains': textAreaallowedEmailDomains 
        })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}


async function createResponseQualityTest() {
    const spinner = document.getElementById('spinnerAdminPanel');
    spinner.style.display = 'block';

    const testsSection = document.getElementById('responseQualityTestCases');
    const newTestDiv = document.createElement('div');
    newTestDiv.className = 'response-quality-test-container';
    newTestDiv.innerHTML = `
        <div style="display: flex; flex-direction: row; width: 100%;">
            <input type="text" placeholder="Query (e.g. 'number of orders')" class="input-with-shadow" style="padding: 8px 10px; margin-bottom: 10px; font-size: 13px; min-width: 100%;">
            <img src="${deleteButtonURL}" class="response-quality-test-delete-button" title='Delete Test' onclick="deleteResponseQualityTest(this)">
        </div>        
        <textarea placeholder="Expected SQL Query (e.g. 'SELECT COUNT(*) FROM orders')" class="input-with-shadow" style="padding: 8px 10px; height: 50px; width:100%; min-width: 100%; "></textarea>
    `;
    testsSection.appendChild(newTestDiv);
    spinner.style.display = 'none';
}


async function deleteResponseQualityTest(button) {
    const testCaseDiv = button.closest('.response-quality-test-container');
    if (testCaseDiv) {
      testCaseDiv.remove();
    }
  }



async function saveResponseQualityTests(dbId) {
    const spinner = document.getElementById('spinnerSaveResponseQualityTests');
    const responseElement = document.getElementById('responseSaveResponseQualityTests');
    responseElement.innerText = '';
    spinner.style.display = 'block';
    
    const testsSection = document.getElementById('responseQualityTestCases');
    const testCases = [];

    for (const testDiv of testsSection.children) {
        const inputElement = testDiv.querySelector('input');
        const textAreaElement = testDiv.querySelector('textarea');
        const query = inputElement.value;
        const expected_sql = textAreaElement.value;
        if (query.trim() != '' ) {
            testCases.push({ query, expected_sql });
        }
    }

    try {
        const response = await fetch('/save-response-quality-tests', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ dbId, dbId, testCases:testCases })
        });
        
        spinner.style.display = 'none';

        const result = await response.text();
        responseElement.innerText = result;
        if (result.includes('Error')) {
            responseElement.style.color = 'red';
        }
        else {
            responseElement.style.color = 'green';
        }
    } catch (error) {
        spinner.style.display = 'none';
        responseElement.innerText = `An error occured while saving: ${error}`;
        responseElement.style.color = 'red'; 
    }
}



function closeTab(tabName) {
    document.getElementById(tabName).classList.remove('active');
}


function logout(isTwoCol) {
    fetch('/logout', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        }
    })
    .then(response => {
        if (response.ok) {
            closePopup();
            window.location.href = isTwoCol === 'True' ? '/login?show-chatbot-and-dashboards=true' : '/login';
        } else {
            alert('Logout failed. Please try again.');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred. Please try again.');
    });
}


///////////////////////////////////////////
// multi-select dropdown (user management)
///////////////////////////////////////////
function toggleDropdown(event) {
    event.stopPropagation();
    const dropdownMenu = event.target.nextElementSibling;
    dropdownMenu.style.display = dropdownMenu.style.display === 'block' ? 'none' : 'block';
}


function toggleCheckmarkGroupMembership(checkbox) {
    const emailAddress = checkbox.getAttribute('emailaddress');
    const groupMembership = checkbox.getAttribute('value');
    

    if (checkbox.checked) {
        checkbox.parentElement.classList.add('checked');
        saveUserGroupMembersip(emailAddress, groupMembership, 'add');
    } else {
        checkbox.parentElement.classList.remove('checked');
        saveUserGroupMembersip(emailAddress, groupMembership, 'remove');
    }
}


function toggleCheckmarkTablePermissions(checkbox) {
    const dbId = checkbox.getAttribute('dbId');
    const tableId = checkbox.getAttribute('tableId');
    const groupMembership = checkbox.getAttribute('value');
    

    if (checkbox.checked) {
        checkbox.parentElement.classList.add('checked');
        saveTablePermission(dbId, tableId, groupMembership, 'add');
    } else {
        checkbox.parentElement.classList.remove('checked');
        saveTablePermission(dbId, tableId, groupMembership, 'remove');
    }
}


// // removed in favor of shift+enter functionality which requires multiple lines and therefore the up/down arrow keys cannot be used for command history
// function addToCommandHistory(userMsg) {
//     if (userMsg && !history.includes(userMsg)) {
//         history.push(userMsg);
//         historyIndex = history.length; // Move to end of history
//     }
// }



async function handleEnter(event) {
    if (event.key === 'Enter' && !event.shiftKey) {
        event.preventDefault(); 
        submitUserMessage(event); // Trigger form submission
    }
}



function showNotification(message, isSuccess, timeout = 2000) {
    const notificationId = 'notification-' + Date.now();
    const notification = document.createElement('div');
    notification.id = notificationId;
    notification.className = 'notification';
    notification.textContent = message;
    notification.style.backgroundColor = isSuccess ? "#44c767" : "#ff4444";
    
    document.body.appendChild(notification);
    
    // Trigger reflow to ensure the transition works
    notification.offsetHeight;
    
    notification.classList.add('show');
    activeNotifications++;
    
    // Position the notification
    notification.style.bottom = (30 + (activeNotifications - 1) * 50) + 'px';
    
    setTimeout(() => {
        notification.classList.remove('show');
        notification.style.bottom = '-100px'; // Slide out to bottom
        
        // Remove the notification after the slide-out animation completes
        notification.addEventListener('transitionend', function handler() {
            document.body.removeChild(notification);
            activeNotifications--;
            notification.removeEventListener('transitionend', handler);
        });
    }, timeout);
}


function startVoiceRecognition() {
    if (!SpeechRecognition) {
        const errorMessage = 'Speech recognition is not supported in this browser. Please use a different browser, or type your query.';
        const conversationDiv = document.getElementById('conversation');
        const errorDiv = document.createElement('div');
        errorDiv.style.textAlign = 'left';
        errorDiv.style.padding = '5px';
        errorDiv.innerHTML = errorMessage;
        conversationDiv.appendChild(errorDiv);
        return;
    }

    userMsgArea = document.getElementById('user_message');
    transcriptionDiv = document.getElementById('transcription');
    transcriptionDiv.textContent = '';  // clear the previous transcription
    
    // Create a new instance of the SpeechRecognition API
    const recognition = new SpeechRecognition();
    recognition.lang = language;
    recognition.interimResults = showInterimResultPref;
    recognition.maxAlternatives = 10;
    
    recognition.onstart = () => {
        listeningDiv.style.visibility = 'visible';
    };

    if (language === 'ar-SA' || language === 'fa-IR' || language === 'he-IL') {
        transcriptionDiv.style.direction = 'rtl';
        userMsgArea.style.direction = 'rtl';
    } else {
        transcriptionDiv.style.direction = 'ltr';
        userMsgArea.style.direction = 'ltr';
    }

    if ( showInterimResultPref == 'True'  ) {
        recognition.onresult = (event) => {
            let transcript = '';
            for (let i = event.resultIndex; i < event.results.length; i++) {
                transcript += event.results[i][0].transcript;
            }
            transcriptionDiv.textContent = transcript.replace(/\.$/, '');
            if (transcriptionDiv.style.direction == 'rtl') {
                transcript = transcript.replace(/\?/g, '؟');
            }
        };
    } else {
        recognition.onresult = (event) => {
            const currentResultIndex = event.resultIndex;
            let transcript = event.results[currentResultIndex][0].transcript.replace(/\.$/, '');
            if (transcriptionDiv.style.direction == 'rtl') {
                transcript = transcript.replace(/\?/g, '؟');
            }
            transcriptionDiv.textContent = transcript;
        };
    }

    recognition.onend = () => {
        recognition.stop();

        setTimeout(() => {
            listeningDiv.style.visibility = 'hidden';
        }, 600);

        userMsgArea.value = transcriptionDiv.textContent;
    };

    recognition.start();
}


function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const sidebarActiveDB = document.getElementById('active-db-container');
    const sidebarToggle = document.getElementById('sidebar-toggle');

    if (sidebar.style.width === '0px') {
        sidebar.style.width = '260px';
        sidebarToggle.style.top = '0px';
        sidebarToggle.style.position = 'relative';
        sidebarActiveDB.style.display = 'block';

        sidebarStatus = 'open';
    } else {
        sidebar.style.width = '0px';
        sidebarToggle.style.position = 'absolute';
        sidebarToggle.style.top = '8px';
        
        sidebarStatus = 'closed';
    }
    
    // adjust the position of status updates
    positionStatusUpdates(statusUpdatesWrapper);
}


async function deleteSession(sessionHash) {
    const response = await fetch('/delete-session', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({'sessionHash': sessionHash })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Session Deletion Failed!", language), false);
        console.log(result);
    } else {
        const session = document.getElementById(`session_${sessionHash}`);
        session.style.transition = 'opacity 0.5s';
        session.style.opacity = '0';
        setTimeout(() => {
            session.style.display = 'none';
        }, 500);
    } 
}



async function loadPersonalDashboard() {
    const personalDashboardPopup = document.getElementById('personalDashboard_BG');
    personalDashboardPopup.style.display = "block";
    const personalDashboardDiv = document.getElementById('personalDashboardSection');
    personalDashboardDiv.innerHTML = '';  // clear the previous content
    try{
        const botResponse = await askBot('show_personal_dashboard');
        if (botResponse.main_response.startsWith('http')) {
            await loadIframeInsideDiv(personalDashboardDiv, botResponse.main_response, 'dashboard');
        } else {
            personalDashboardDiv.innerHTML = `
            <button class="close-button" onclick="closePopupScreen('personalDashboard_BG')">
                <img src="${closeIconUrl}" class="close_icon">Close
            </button>
            <button style="margin: auto; cursor: pointer; position: absolute; top: 30px; padding: 5px; width: 120px; left: calc(50% - 60px);" onclick="createTab()">
                Create a Tab
            </button>
            <div id="personalDashboardSpinner" class="spinner" style="display: none; position: absolute; top: 70px; padding: 5px; width: 10px; height: 10px; left: calc(50% - 15px);">
            </div>
            <p style="margin: 100px 20px;">
                Your personal dashboard is empty.<br> You can ask questions and add the results to your dashboard.
            </p>`;
        }
    } catch (error) {
        personalDashboardDiv.innerHTML = `
        <button class="close-button" onclick="closePopupScreen('personalDashboard_BG')">
            <img src="${closeIconUrl}" class="close_icon">Close
        </button>
        <p style="margin: 100px 20px;">
            An error occurred while loading the personal dashboard. (${error.stack})
        </p>`;
    }

}

async function createTab() {
    const spinner = document.getElementById('personalDashboardSpinner');
    try {

        spinner.style.display = 'block'; 
        const botResponse = await askBot('create a tab in dashboard');
        spinner.style.display = 'none';  

        if (botResponse.main_response.startsWith('Error')) {
            showNotification(get_translation('Tab Creation Failed!', language), false);
            console.log("An error occurred while creating the tab: " + botResponse.main_response);
        } else {
            showNotification(get_translation('Tab Created Successfully.', language), true);
            await loadPersonalDashboard();
        }

    } catch (error) {
        spinner.style.display = 'none';
        console.log("An error occurred: " + error.stack);
    }    
}


async function toggleChatbot() {
    
    const isHidden = rightColumn.classList.contains('hidden');

    if (isHidden) { // Show panel
      
      rightColumn.classList.remove('hidden');
      divider.classList.remove('hidden');

      // Restore previous width or set default
      let savedWidth = localStorage.getItem('leftColumnWidth');
      if (!savedWidth) {
        savedWidth = '72%';
      } else {
        savedWidth += 'px';
      }
      leftColumn.style.width = savedWidth;
      localStorage.setItem('rightPanelVisible', 'true');

    } else {  // Hide panel

      rightColumn.classList.add('hidden');
      divider.classList.add('hidden');
      leftColumn.style.width = '100%';
      localStorage.setItem('rightPanelVisible', 'false');
      
    }
}


async function loadDashboard(dashboardId) {
    const leftColumnDashboardIframe = document.getElementById('left-column-dashboard-iframe');
    
    // get embedding url for the dashboard
    const response = await fetch('/get-left-column-dashboard-embedding-url', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({'dashboardId': dashboardId })
    });
    const result = await response.text();
    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Failed to load the dashboard!", language), false);
        console.log(result);
    } else {
        leftColumnDashboardIframe.src = result;
    } 
    
}


async function startNewSession() {
    statusUpdatesDiv.innerHTML = '';  // since we show the spinner, we need to clear the previous status updates
    await clearConversationHistory(true, true);
    await loadGreetingsAndQuerySuggestions();
}


async function makeViz(chartConfigString) {
    const conversationDiv = document.getElementById('conversation');

    try {
        const canvas = document.createElement('canvas');
        canvas.id = 'customChart_' + performance.now();  // Date.now(); is not unique enough
        canvas.style.width = '100%';
        conversationDiv.appendChild(canvas);

        const chartConfig = (new Function('return ' + chartConfigString))();  // convert string to object
    
        const ctx = document.getElementById(canvas.id).getContext('2d');
        const customChart = new Chart(ctx, chartConfig);
    
    } catch (error) {
        const errorDiv = document.createElement('div');
        errorDiv.style.textAlign = 'left';
        errorDiv.style.padding = '5px';
        errorDiv.innerHTML = `An error occurred while creating the chart: ${error}`;
        conversationDiv.appendChild(errorDiv);
    }
    
}


async function markAsBadResponse() {
    const response = await fetch('/mark-as-bad-response', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ })
    });
    const result = await response.text();

    if (result.startsWith('Error') ) {   
        showNotification(get_translation("Save failed!", language), false);
        console.log(result);
    } else {
        showNotification(get_translation("Saved.", language), true);
    } 
}



async function downloadPDF(responseText, pdfTitle) {

    const res = await fetch('/export-pdf', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
            content: responseText,
            title: pdfTitle || 'Chatbot Response',
        })
    });

    if (!res.ok) {
        showNotification(get_translation('Failed to generate PDF', language), false);
        console.error('Failed to generate PDF:', res.statusText);
        return; 
    }

    const blob = await res.blob();
    const url = window.URL.createObjectURL(blob);

    const a = document.createElement('a');
    a.href = url;
    a.download = 'Chatbot_Response.pdf';
    document.body.appendChild(a);
    a.click();
    a.remove();
    window.URL.revokeObjectURL(url);
}


function positionStatusUpdates(statusUpdatesWrapper) {
    const sidebar = document.getElementById("sidebar");
    const contentWrapper = document.getElementById("content-wrapper");

    if (!sidebar || !contentWrapper) return;

    const updatePosition = () => {
        const sidebarRect = sidebar.getBoundingClientRect();
        const contentWrapperRect = contentWrapper.getBoundingClientRect();

        // Calculate left: sidebar width + half of main width (relative to sidebar's left)
        const left = sidebarRect.width + contentWrapperRect.width / 2;
        statusUpdatesWrapper.style.left = `${left}px`;
        statusUpdatesWrapper.style.transform = "translate(-175px, -50%)";
    };

    updatePosition();

    // If the sidebar is transitioning, wait for the transition to end before updating position again
    if (getComputedStyle(sidebar).transitionDuration !== "0s") {
        const handler = () => {
            updatePosition();
            sidebar.removeEventListener("transitionend", handler);
        };
        sidebar.addEventListener("transitionend", handler);
    }
}


async function saveFileUploadConfig() {
    
    const spinner = document.getElementById('spinnerSaveUploadFile');
    const responseElement = document.getElementById('responseSaveUploadFileConfig');
    responseElement.innerText = '';
    spinner.style.display = 'block';

    const useUploadedFilesAsDataSource = document.getElementById('toggle-button-use-uploaded-files-as-data-source').checked;
    const vectorStoreId = document.getElementById('vectorStoreId').value;

    const response = await fetch('/save-file-upload-config', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ useUploadedFilesAsDataSource: useUploadedFilesAsDataSource, vectorStoreId: vectorStoreId })
    });
    const result = await response.text();
    spinner.style.display = 'none';
    responseElement.innerText = result;
    if (result.startsWith('Error')) {
        responseElement.style.color = 'red';
    }
    else {
        responseElement.style.color = 'green';
    }
}

// function displaySelectedFiles() {
//     const input = document.getElementById('file-upload-input');
//     const listDiv = document.getElementById('selected-files-list');
//     const files = input.files;
//     if (!files.length) {
//         listDiv.innerHTML = '';
//         return;
//     }
//     let html = '<ul style="padding-left: 40px; text-align: left;">';
//     for (var i = 0; i < files.length; i++) {
//         html += '<li>' + files[i].name + ' (' + Math.round(files[i].size/1024) + ' KB)</li>';
//     }
//     html += '</ul>';
//     listDiv.innerHTML = html;

//     const uploadButton = document.getElementById('upload-btn');
//     if (files.length > 0) {
//         uploadButton.style.display = 'inline-block';
//     } else {
//         uploadButton.style.display = 'none';
//     }
// }



// async function uploadFiles() {
//     const input = document.getElementById('file-upload-input');
//     const files = input.files;
    
//     const formData = new FormData();
//     for (let i = 0; i < files.length; i++) {
//         formData.append('files', files[i]); 
//     }

//     fetch('/upload-files', {
//         method: 'POST',
//         body: formData
//     }).catch(error => {
//         console.error('Upload failed:', error);
//     });
    
// }



let socket; 
async function connectWebSocket() {
    const protocol = window.location.protocol === "https:" ? "wss" : "ws";
    const host = window.location.host;
    
    socket = new WebSocket(`${protocol}://${host}/ws`);

    socket.onopen = function () {
        //console.log("WebSocket connected");
    };
    
    let htmlBuffer = "";
    let currentResponseDivForStreaming = null;

    socket.onmessage = async function (event) {
        const botResponse = JSON.parse(event.data);
        
        if (botResponse.ws_msg_type === "status_update") {
            
            const statusUpdate = document.createElement('div');
            statusUpdate.textContent = botResponse.status;
            statusUpdatesDiv.appendChild(statusUpdate);

        } else if (botResponse.ws_msg_type === "final_response") {
            
            loadingOverlay.style.display = 'none';

            if (botResponse.explanation) {
                await handleBotResponse(botResponse.main_response, false, botResponse.viz_type, botResponse.is_single_color);
                await handleBotResponse(botResponse.explanation);
            } else {
                // main operation
                try {
                    await handleBotResponse(botResponse.main_response, false, botResponse.viz_type, botResponse.is_single_color);
                }
                catch (error) {
                    console.error(error);
                    const errorDiv = document.createElement('div');
                    errorDiv.style.textAlign = 'left';
                    errorDiv.style.padding = '5px';
                    errorDiv.innerHTML = `An error occurred while processing the response: ${error}`;
                    conversationDiv.appendChild(errorDiv);
                }
            }
            await scrollToBottom();
        
        } else if (botResponse.ws_msg_type === "response_stream") {
            
            loadingOverlay.style.display = 'none';  // hide the status updates screen

            if (botResponse.stream_data_type === "delete_div") {
                // delete the div that was created for streaming
                if (currentResponseDivForStreaming) {
                    currentResponseDivForStreaming.remove();
                    currentResponseDivForStreaming = null;
                }
                htmlBuffer = "";  
                loadingOverlay.style.display = 'block'; // show the loading overlay again for viz status update

            } else if (botResponse.stream_data_type === "chunk") {
                
                if (!currentResponseDivForStreaming) {  // means streaming has just started
                    
                    // create a new div to show the streaming response
                    currentResponseDivForStreaming = document.createElement('div');
                    if ( language === 'ar-SA' || language === 'fa-IR' || language === 'he-IL' ) {
                        currentResponseDivForStreaming.className = 'bot-response-rtl';
                    } else {
                        currentResponseDivForStreaming.className = 'bot-response-ltr';
                    }
                    currentResponseDivForStreaming.classList.add('bot-response');
                    
                    // add div to UI
                    conversationDiv.appendChild(currentResponseDivForStreaming);
                }

                htmlBuffer += botResponse.stream_data_content;
                
                // replace newlines with <br> but not inside <table> tags
                let formattedResponse;
                const tableStartIndex = htmlBuffer.search(/<table[\s\S]*?>/i);
                const tableEndIndex = htmlBuffer.search(/<\/table>/i);
                if (tableStartIndex !== -1 && (tableEndIndex === -1 || tableEndIndex < tableStartIndex)) {
                    // Incomplete table: don't touch newlines inside the table
                    // Only format text before <table>
                    const beforeTable = htmlBuffer.slice(0, tableStartIndex)
                        .replace(/\r/g, '')
                        .replace(/\n/g, '<br>')
                        .replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');
                    const afterTable = htmlBuffer.slice(tableStartIndex); // leave as-is
                    formattedResponse = beforeTable + afterTable;
                } else if (htmlBuffer.includes('</table>')) {
                    // Table is complete: apply selective formatting
                    const parts = htmlBuffer.split(/(<table[\s\S]*?<\/table>)/i);
                    for (let i = 0; i < parts.length; i++) {
                        if (!parts[i].toLowerCase().startsWith('<table')) {
                            parts[i] = parts[i]
                                .replace(/\n/g, '<br>')
                                .replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');
                        }
                    }
                    formattedResponse = parts.join('');
                } else {
                    // No table: format everything
                    formattedResponse = htmlBuffer
                        .replace(/\r/g, '')
                        .replace(/\n/g, '<br>')
                        .replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;');
                }
                currentResponseDivForStreaming.innerHTML = formattedResponse;
            }

            else if (botResponse.stream_data_type === "done") {  // end of stream
                
                if (isAnalyticalMode && isSession == false) {
                    await addActionButtonsAnalyticalMode(currentResponseDivForStreaming);
                }
                
                currentResponseDivForStreaming = null;
                htmlBuffer = "";

                await scrollToBottom();
            }

        } else if (botResponse.ws_msg_type === "test_run_result") {
            
            const testIndex = botResponse.test_index;
            const testResultsRow = document.getElementById('test_run_result_tr_' + testIndex);
            const testResult = botResponse.main_response;
            const stopSpinnerFlag = botResponse.stop_spinner;
            
            if ( stopSpinnerFlag == true ) {
                const spinnerTestRunResults = document.getElementById('spinnerTestRunResults');
                spinnerTestRunResults.style.visibility = 'hidden';
            }

            if (testResult.startsWith('Running')) {
                testResultsRow.cells[1].style.color = 'blue';
                testResultsRow.cells[1].innerHTML = testResult;
            } else if (testResult.startsWith('Passed')) {
                testResultsRow.cells[1].style.color = 'green';
                testResultsRow.cells[1].innerHTML = testResult;
            } else {
                const error = botResponse.error;
                testResultsRow.cells[1].style.color = 'red';
                testResultsRow.cells[1].innerHTML = `<a href="#" style="color: red;">${testResult}</a>`;
                testResultsRow.cells[1].onclick = function() {
                    showFailedTestError(error);
                };
            }
            
        } else {
            loadingOverlay.style.display = 'none';
            handleBotResponse('Unexpected Websocket message type: ' + botResponse.ws_msg_type);
        }
        
    };

    socket.onclose = function () {
        console.log("WebSocket closed, reconnecting...");
        setTimeout(connectWebSocket, 5000);  // Attempt to reconnect after 5s
    };

    socket.onerror = function (error) {
        console.error("WebSocket error:", error);
    };
}

connectWebSocket();